USE hotels;

CREATE TABLE IF NOT EXISTS _sequence
(
  seq_name VARCHAR(50)  NOT NULL PRIMARY KEY,
  seq_val  INT UNSIGNED NOT NULL
);

CREATE TABLE IF NOT EXISTS Stars (
  id   INT PRIMARY KEY,
  name VARCHAR(100) NOT NULL
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS Partners (
  id   INT PRIMARY KEY,
  name VARCHAR(100) NOT NULL
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS Hotels (
  id               INT PRIMARY KEY,
  geo_id           INT DEFAULT 10000,
  longitude        DOUBLE        NOT NULL,
  latitude         DOUBLE        NOT NULL,
  partner_id       INT           NOT NULL REFERENCES Partners (id),
  partner_inner_id VARCHAR(200)  NOT NULL,
  address_info     VARCHAR(2000) NOT NULL,
  name_info        VARCHAR(600)  NOT NULL,
  url              VARCHAR(1024) NULL,
  partner_url      VARCHAR(1024) NULL,
  stars_id         INT           NOT NULL REFERENCES Stars (id),
  cluster_id       INT           NOT NULL REFERENCES Hotels (id),
  update_ts        TIMESTAMP     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  updated_by       VARCHAR(20)   NOT NULL DEFAULT 'auto',
  history_id       INT           NOT NULL REFERENCES Hotels (id),
  features         TEXT          NULL,
  FULLTEXT name_ft (name_info),
  FULLTEXT feature_ft (features),
  FULLTEXT address_ft (address_info),
  INDEX geo_id_index (geo_id),
  INDEX cluster_id_index (cluster_id),
  INDEX history_id_index (history_id),
  INDEX partner_index (partner_id, partner_inner_id)
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS GeoIndex (
  longitude INT NOT NULL,
  latitude  INT NOT NULL,
  hotel_id  INT NOT NULL,
  FOREIGN KEY (hotel_id) REFERENCES Hotels(id),
  INDEX geo_index (longitude, latitude)
);

CREATE TABLE IF NOT EXISTS ManualClustering (
  id         INT PRIMARY KEY AUTO_INCREMENT,
  first      INT         NOT NULL REFERENCES Hotels (id),
  second     INT         NOT NULL REFERENCES Hotels (id),
  is_similar BOOLEAN     NOT NULL,
  updated_by VARCHAR(20) NOT NULL DEFAULT 'auto',
  update_ts  TIMESTAMP   NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS RecallSamples (
  id         INT PRIMARY KEY AUTO_INCREMENT,
  sample_id  INT         NOT NULL,
  hotel_id   INT         NOT NULL REFERENCES Hotels (id),
  status     VARCHAR(10) NOT NULL DEFAULT 'NEW',
  updated_by VARCHAR(20) NOT NULL DEFAULT 'auto',
  update_ts  TIMESTAMP   NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS GrayClustering (
  id         INT PRIMARY KEY AUTO_INCREMENT,
  hotel_id   INT         NOT NULL REFERENCES Hotels (id),
  update_ts  TIMESTAMP   NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  updated_by VARCHAR(20) NOT NULL DEFAULT 'auto',
  status     VARCHAR(10) NOT NULL DEFAULT 'NEW'
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS ConflictsInCluster (
  id         INT PRIMARY KEY AUTO_INCREMENT,
  conflicts  INT           NOT NULL,
  cluster_id INT           NOT NULL REFERENCES Hotels (id),
  fields     VARCHAR(1000) NOT NULL,
  update_ts  TIMESTAMP     NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  updated_by VARCHAR(20)   NOT NULL DEFAULT 'auto',
  status     VARCHAR(10)   NOT NULL DEFAULT 'NEW'
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

CREATE TABLE IF NOT EXISTS PrecisionSamples (
  id         INT PRIMARY KEY AUTO_INCREMENT,
  sample_id  INT         NOT NULL,
  hotel_id   INT         NOT NULL REFERENCES Hotels (id),
  status     VARCHAR(10) NOT NULL DEFAULT 'NEW',
  updated_by VARCHAR(20) NOT NULL DEFAULT 'auto',
  update_ts  TIMESTAMP   NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
  ENGINE = InnoDB
  DEFAULT CHARSET = utf8;

DELIMITER //
DROP FUNCTION IF EXISTS getNextSeq//

CREATE FUNCTION getNextSeq(sSeqName VARCHAR(50))
  RETURNS INT UNSIGNED
  BEGIN
    DECLARE nLast_val INT;

    SET nLast_val = (SELECT
                       seq_val
                     FROM _sequence
                     WHERE seq_name = sSeqName);
    IF nLast_val IS NULL
    THEN
      SET nLast_val = 1;
      INSERT INTO _sequence (seq_name, seq_val)
      VALUES (sSeqName, nLast_Val);
    ELSE
      SET nLast_val = nLast_val + 1;
      UPDATE _sequence
      SET seq_val = nLast_val
      WHERE seq_name = sSeqName;
    END IF;

    RETURN nLast_val;
  END//

DROP PROCEDURE IF EXISTS sp_setSeqVal//

CREATE PROCEDURE sp_setSeqVal(sSeqName VARCHAR(50), nVal INT UNSIGNED)
  BEGIN
    IF (SELECT
          count(*)
        FROM _sequence
        WHERE seq_name = sSeqName) = 0
    THEN
      INSERT INTO _sequence (seq_name, seq_val)
      VALUES (sSeqName, nVal);
    ELSE
      UPDATE _sequence
      SET seq_val = nVal
      WHERE seq_name = sSeqName;
    END IF;
  END//

CREATE TRIGGER on_insert_hotel AFTER INSERT ON Hotels
FOR EACH ROW
  BEGIN
    IF NEW.history_id = new.id
    THEN INSERT INTO GeoIndex (longitude, latitude, hotel_id) VALUES (truncate(NEW.longitude, 0), truncate(NEW.latitude, 0), NEW.id);
    END IF;
  END//

CREATE TRIGGER on_update_hotel AFTER UPDATE ON Hotels
FOR EACH ROW
  BEGIN
    IF NEW.history_id = new.id AND OLD.history_id != OLD.id
    THEN INSERT INTO GeoIndex (longitude, latitude, hotel_id) VALUES (truncate(NEW.longitude, 0), truncate(NEW.latitude, 0), NEW.id);
    ELSEIF OLD.history_id = OLD.id AND NEW.history_id != NEW.id
      THEN DELETE FROM GeoIndex
      WHERE hotel_id = OLD.id;
    END IF;
  END//

CREATE TRIGGER on_delete_hotel AFTER DELETE ON Hotels
FOR EACH ROW
  BEGIN
    IF OLD.history_id = OLD.id
    THEN DELETE FROM GeoIndex
    WHERE hotel_id = OLD.id;
    END IF;
  END//

DELIMITER ;