package views

import models.Hotel
import org.joda.time.DateTime
import org.joda.time.format.{DateTimeFormat, DateTimeFormatterBuilder}
import play.api.Play
import ru.yandex.tours.app.CommonDataHolders
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db.model.DbPartnerHotel
import ru.yandex.tours.direction.DirectionsStats
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.geo.partners.{PartnerRegion, PartnerTrees, PartnerTreesSupport}
import ru.yandex.tours.hotels.enrichers.{GeoIdSettersSupport, MetaGeoIdSetter}
import ru.yandex.tours.hotels.{HotelRatings, HotelsIndex}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Features.Feature
import ru.yandex.tours.model.hotels.HotelsHolder.{PartnerHotel, RawPartnerHotel}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.tanker.Translations

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.10.15
 */
object Utils {

  private var _application: CommonDataHolders with PartnerTreesSupport with GeoIdSettersSupport = null
  def init(application: CommonDataHolders with PartnerTreesSupport with GeoIdSettersSupport): Unit = {
    _application = application
  }

  val frontHost = Play.current.configuration.getString("tours.front.host").getOrElse("travel.yandex.ru")
  val dateFormat = new DateTimeFormatterBuilder()
    .appendDayOfMonth(1)
    .appendLiteral(' ')
    .appendMonthOfYearText()
    .appendLiteral(' ')
    .appendYear(1, 4)
    .toFormatter

  val dateTimeFormat = DateTimeFormat.fullDateTime()

  def printDateTime(ts: Long): String = Utils.dateTimeFormat.print(new DateTime(ts))
  def printDateTime(dateTime: DateTime): String = Utils.dateTimeFormat.print(dateTime)

  def tree: Tree = _application.regionTree
  def partnerTrees: PartnerTrees = _application.partnerTrees
  def hotelsIndex: HotelsIndex = _application.hotelsIndex
  def hotelRatings: HotelRatings = _application.hotelRatings
  def geoMappings: GeoMappingHolder = _application.geoMappingHolder
  def directionStats: DirectionsStats = _application.directionsStats
  def geoIdSetter: MetaGeoIdSetter = _application.metaGeoIdSetter
  def translations: Translations = _application.translations

  def regionName(geoId: Int): String = tree.region(geoId).fold("Неизвестный регион")(_.name.ruName)
  def hotelName(hotel: Hotel): String = {
    val stars = if (hotel.star.id > 0) " " + hotel.star.id + "*" else ""
    hotel.name.ruName + stars
  }

  def hotelName(hotel: PartnerHotel): String = {
    val stars = if (hotel.getRawHotel.hasStars) " " + hotel.getRawHotel.getStars + "*" else ""
    hotel.getRawHotel.getNameList.asScala.headOption.map(_.getValue).getOrElse("") + stars
  }

  def regionPath(geoId: Int): List[Region] = {
    val res = tree.pathToRoot(geoId).filter(r => geoMappings.isKnownDestination(r.id)).reverse
    if (res.isEmpty) tree.region(geoId).toList
    else res
  }

  def hotelServiceLink(hotelId: Int): String = {
    s"https://$frontHost/hotel/$hotelId"
  }

  def featureName(feature: String): String = {
    translations.translateFeature(feature, Languages.ru).fold(feature) {
      tr => s"$feature – $tr"
    }
  }

  def traverseTree(regionId: Int, all: Boolean): Traversable[(Region, Int)] = {
    new Traversable[(Region, Int)] {
      private def isKnown(region: Region) = {
        all || geoMappings.isKnownDestination(region.id) || tree.root.contains(region)
      }

      private def traverse[U](region: Region, level: Int, f: ((Region, Int)) => U): Unit = {
        if (isKnown(region)) f((region, level))
        else {
          for (child <- tree.children(region)) {
            traverse(child, level, f)
          }
        }
      }

      override def foreach[U](f: ((Region, Int)) => U): Unit = {
        val path = tree.pathToRoot(regionId).filter(isKnown).toVector
        path.reverse.zipWithIndex.foreach(f)
        tree.children(regionId).toSeq.filter(isKnown).sortBy(-_.position).foreach(r => traverse(r, path.size, f))
      }
    }
  }

  def getPartnerRegion(rawHotel: RawPartnerHotel): Option[PartnerRegion] = {
    val partner = Partners(rawHotel.getPartner)
    for {
      partnerRegion <- if (rawHotel.hasRegionId) Some(rawHotel.getRegionId) else None
      tree <- partnerTrees.get(partner)
      region <- tree.region(partnerRegion)
    } yield region
  }
}
