package ru.yandex.tours

import akka.actor.Props
import akka.routing.RoundRobinPool
import ru.yandex.tours.agencies._
import ru.yandex.tours.api.ReqAnsLogger
import ru.yandex.tours.api.v1._
import ru.yandex.tours.api.v1.hotel.{GetOfferRouter, HotelHandler, OfferSearchRouter}
import ru.yandex.tours.api.v1.recommend.{RecommendHandler, RootRecommendHandler}
import ru.yandex.tours.api.v1.reference.{OtaHandler, ReferenceHandler}
import ru.yandex.tours.api.v1.resort.ResortSkiHandler
import ru.yandex.tours.api.v1.search.{FlightSearchHandler, RootSearchHandler, SearchHandler, TransferSearchHandler}
import ru.yandex.tours.api.v1.statistic.{RootStatisticHandler, StatisticHandler}
import ru.yandex.tours.api.v1.subscription.SubscriptionsHandler
import ru.yandex.tours.api.v1.users.UsersHandler
import ru.yandex.tours.app._
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.backend._
import ru.yandex.tours.billing.BillingComponents
import ru.yandex.tours.direction.CountryResortsService
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.filter.{HiddenFilters, SnippetFiltrator}
import ru.yandex.tours.hotels._
import ru.yandex.tours.metro.MetroHolder
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.{HotelProvider, TourOperator}
import ru.yandex.tours.operators.{TourOperatorAvailability, TourOperatorsState}
import ru.yandex.tours.ota.{AuctionService, OnlineTourAgencies, OtaMappingHolder, PartnerStatisticHolder}
import ru.yandex.tours.services._
import ru.yandex.tours.util.spray.{PingControl, RouteesContext}
import ru.yandex.tours.util.LabelBuilder
import shapeless._

import scala.concurrent.duration._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.01.15
 */
trait ApiComponents {
  self: Application
    with AkkaSupport with SpraySupport
    with CommonDataHolders with ExtDataSupport
    with HttpSupport
    with CalendarServiceSupport
    with UserServicesSupport
    with BillingComponents
    with AviaClientSupport =>

  import akkaSystem.dispatcher

  private implicit val actorSystem = akkaSystem

  lazy val otaMappingHolder = OtaMappingHolder.from(extDataService, extDataUpdateChecker)
  lazy val agenciesBillingIndex = AgenciesBillingIndex.from(HNil, extDataService, extDataUpdateChecker)
  lazy val hotelsBillingIndex = HotelsBillingIndex.from(hotelsIndex :: HNil, extDataService, extDataUpdateChecker)
  lazy val cmHotelsBillingIndex = CMHotelsBillingIndex.from(hotelsIndex :: HNil, extDataService, extDataUpdateChecker)
  lazy val partnerStatisticHolder = PartnerStatisticHolder.from(extDataService, extDataUpdateChecker)
  lazy val auctionService = new AuctionService(partnerStatisticHolder)
  lazy val agenciesPartneringIndex = AgencyPartneringIndex.from(extDataService, extDataUpdateChecker)
  lazy val customer2operatorIndex = CustomerToOperatorIndex.from(extDataService, extDataUpdateChecker)
  lazy val agencies = Agencies.from(extDataService, extDataUpdateChecker)
  lazy val agenciesIndex = AgenciesIndex.from(
    agencies :: regionTree :: agenciesBillingIndex :: agenciesPartneringIndex :: customer2operatorIndex :: HNil,
    extDataUpdateChecker
  )

  lazy val metroHolder = MetroHolder.from(extDataService, extDataUpdateChecker)
  lazy val tourOperatorAvailability = new TourOperatorAvailability(
    config.getString("tours.hydra.limiter.host"),
    config.getInt("tours.hydra.limiter.port"),
    httpClient
  )
  lazy val tourOperatorState = new TourOperatorsState(tourOperators, tourOperatorAvailability, 30.seconds)
  lazy val onlineAgencies = OnlineTourAgencies.from(extDataService, extDataUpdateChecker)

  lazy val routeesContext = new RouteesContext(departures, regionTree, Some(calendarService))

  val reqAnsLogger = new ReqAnsLogger(config.getString("tours.reqans.log.format"))

  val minPriceConfig = config.getConfig("calendar")

  val toursMinPriceService = RemoteMinPriceService.fromConfig(httpClient, minPriceConfig, "tours")
  val roomsMinPriceService = RemoteMinPriceService.fromConfig(httpClient, minPriceConfig, "rooms")
  val toursSearcher = RemoteToursSearchService.fromConfig(httpClient, config, "tours")
  val roomsSearcher = RemoteHotelSearchService.fromConfig(httpClient, config, "rooms")
  val flightSearchService = RemoteFlightSearchService.fromConfig(httpClient, config)
  val transferSearchService = RemoteTransferSearchService.fromConfig(httpClient, config)

  val hiddenFilters = HiddenFilters.from(extDataService, extDataUpdateChecker)
  val snippetFiltrator = SnippetFiltrator.from(hotelsIndex :: hiddenFilters :: HNil, extDataUpdateChecker)
  val countryResortService = new CountryResortsService(directions, hotelsIndex, regionTree, geoMappingHolder)

  val toursRecommendService = RemoteRecommendService.fromConfig(httpClient, config, "tours")
  val roomsRecommendService = RemoteRecommendService.fromConfig(httpClient, config, "rooms")

  val toursHotelSnippetPreparer = new OfferPreparer(toursSearcher, hotelsIndex, hotelsSimilarity,
    regionTree, snippetFiltrator, billingService, hotelsBillingIndex, cmHotelsBillingIndex, tourOperators, hotelRatings,
    searchSettings) with SearchSettingsSupport[TourOperator]
  val roomsHotelSnippetPreparer = new HotelSnippetPreparer(roomsSearcher, hotelsIndex, hotelsSimilarity,
    regionTree, snippetFiltrator, hotelRatings, billingService, hotelsBillingIndex, cmHotelsBillingIndex,
    hotelProviders, SearchType.ROOMS, searchSettings) with SearchSettingsSupport[HotelProvider]
  val minPricePreparer = new HotelMinPricePreparer(toursHotelSnippetPreparer, roomsHotelSnippetPreparer)

  val redirectUrl = Try(config.getString("tours.redirect.url")).getOrElse("")
  val labelBuilder = new LabelBuilder(config.getString("tours.label.key"))

  val toursSearchHandler = akkaSystem.actorOf(Props(
    new SearchHandler(routeesContext, toursHotelSnippetPreparer, minPricePreparer,
      regionTree, geoMappingHolder, searchSettings,
      hotelsIndex, departures, tourOperators,
      cacheMetricName = "tours_hotel_search",
      SearchType.TOURS, hotelsVideo, reqAnsLogger, redirectUrl, labelBuilder)
  ), "tours-search-handler")
  val roomsSearchHandler = akkaSystem.actorOf(Props(
    new SearchHandler(routeesContext, roomsHotelSnippetPreparer, minPricePreparer,
      regionTree, geoMappingHolder, searchSettings,
      hotelsIndex, departures, hotelProviders,
      cacheMetricName = "rooms_hotel_search",
      SearchType.ROOMS, hotelsVideo, reqAnsLogger, redirectUrl, labelBuilder)
  ), "rooms-search-handler")

  val flightSearchHandler = akkaSystem.actorOf(
    FlightSearchHandler.props(routeesContext, flightSearchService,
      airportRecommendations, departures, airports, regionTree, reqAnsLogger),
    name = "flight-search-handler"
  )

  val transferSearchHandler = akkaSystem.actorOf(
    TransferSearchHandler.props(routeesContext, transferSearchService, airportRecommendations,
      regionTree, airports, hotelsIndex)
  )

  val searchHandler = akkaSystem.actorOf(
    Props(new RootSearchHandler(
      toursSearchHandler,
      roomsSearchHandler,
      flightSearchHandler,
      transferSearchHandler)),
    name = "v1-search-handler"
  )

  val roomsRecommendHandler = akkaSystem.actorOf(Props(
    new RecommendHandler(regionTree, roomsRecommendService, SearchType.ROOMS)
  ), "rooms-recommend-handler")
  val toursRecommendHandler = akkaSystem.actorOf(Props(
    new RecommendHandler(regionTree, toursRecommendService, SearchType.TOURS)
  ), "tours-recommend-handler")

  val recommendHandler = akkaSystem.actorOf(Props(
    new RootRecommendHandler(
      routeesContext,
      regionTree,
      hotelsIndex, hotelsVideo,
      geoMappingHolder,
      searchSettings,
      directions, directionsStats,
      airportRecommendations,
      roomsRecommendHandler, toursRecommendHandler)
  ), "recommend-handler")

  val toursStatisticHandler = akkaSystem.actorOf(Props(
    new StatisticHandler(routeesContext, toursMinPriceService, urlBuilder, regionTree, calendarService,
      tourOperators, SearchType.TOURS)
  ), "tours-statistic-handler")
  val roomsStatisticHandler = akkaSystem.actorOf(Props(
    new StatisticHandler(routeesContext, roomsMinPriceService, urlBuilder, regionTree, calendarService,
      hotelProviders, SearchType.ROOMS)
  ), "rooms-statistic-handler")
  val statisticHandler = akkaSystem.actorOf(Props(
    new RootStatisticHandler(roomsStatisticHandler, toursStatisticHandler,
      routeesContext, calendarService, tourOperatorState)
  ), "statistic-handler")

  val hotelsExport = new HotelsExport(extDataService, extDataUpdateChecker)
  val hotelsExportMarshaller = new HotelsExportMarshaller(hotelsExport)(akkaSystem)
  val otaHandler = akkaSystem.actorOf(Props(
    new OtaHandler(geoMappingHolder, regionTree,
      hotelsIndex, hotelsVideo, hotelsExport, hotelsExportMarshaller,
      tourOperators, onlineAgencies)
  ), "ota-handler")
  val referenceHandler = akkaSystem.actorOf(Props(
    new ReferenceHandler(regionTree, geoMappingHolder, tourOperators, hotelProviders, onlineAgencies, hotelsIndex,
      metroHolder, calendarService, countryResortService, agenciesIndex, brandings, hotelsExportMarshaller, otaHandler)
  ), "reference-handler")

  val subscriptionsHandler = akkaSystem.actorOf(Props(new SubscriptionsHandler), "subscriptions-handler")

  val resortSkiHandler = akkaSystem.actorOf(Props(new ResortSkiHandler(
    skiResorts, directions, currencyRates, regionTree, extDataService
  )), "v1-resort-ski-handler")

  val usersHandler = akkaSystem.actorOf(Props(new UsersHandler(
    routeesContext,
    hotelsIndex, hotelsVideo, regionTree, geoMappingHolder, searchSettings,
    userHistoryService, userFavoritesService
  )), "v1-users-handler")

  val purchaseService = new PurchaseService(billingService, otaMappingHolder, oldLtGeoMappingHolder,
    hotelsIndex, regionTree, tourOperators, onlineAgencies, auctionService)

  val roomsOfferSearchRouter = new OfferSearchRouter(routeesContext, roomsHotelSnippetPreparer, regionTree,
    hotelProviders, SearchType.ROOMS, geoMappingHolder, searchSettings, hotelsVideo,
    reqAnsLogger, redirectUrl, labelBuilder)(akkaSystem.dispatcher)

  val toursOfferSearchRouter = new GetOfferRouter(routeesContext, toursHotelSnippetPreparer, purchaseService,
    geoMappingHolder, searchSettings, regionTree, tourOperators, SearchType.TOURS, hotelsVideo,
    reqAnsLogger, redirectUrl, labelBuilder)(akkaSystem.dispatcher)

  val hotelHandler = {
    val props = Props(new HotelHandler(
      routeesContext,
      hotelsIndex,
      regionTree,
      geoMappingHolder,
      searchSettings,
      directionsStats,
      toursOfferSearchRouter.route,
      roomsOfferSearchRouter.route,
      minPricePreparer,
      hotelsVideo,
      hotelsPanoramas,
      airportRecommendations,
      skiResorts
    )).withRouter(RoundRobinPool(nrOfInstances = 8))
    akkaSystem.actorOf(props, "hotel-handler")
  }
  val agenciesHandler = akkaSystem.actorOf(Props(
    new AgenciesHandler(agencies, agenciesIndex, metroHolder)
  ), "agencies-handler")

  val handlerProps = Props(new ApiHandler(
    searchHandler,
    recommendHandler,
    statisticHandler,
    hotelHandler,
    agenciesHandler,
    referenceHandler,
    subscriptionsHandler,
    resortSkiHandler,
    usersHandler
  ))

  val pingControl = {
    val pinger = new PingControl
    pinger.scheduleRemoteCheck(config.getConfig("searcher"), httpClient, 3.seconds, 500.millis)
    pinger.scheduleRemoteCheck(config.getConfig("calendar"), httpClient, 3.seconds, 500.millis)
    pinger
  }

  onStart {
    geoMappingHolder
    regionTree
    hotelsIndex
    directions
    departures
    otaMappingHolder
    agenciesIndex
    AgenciesBillingIndex.monitor(agenciesBillingIndex)

    startServer(handlerProps, config.getInt("component.port"), pingControl)
  }
}
