package ru.yandex.tours.api

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.backend.HotelMinPrice
import ru.yandex.tours.backend.HotelSnippetPreparer.{OfferStatistic, SnippetWithBilling}
import ru.yandex.tours.billing.BillingOffers
import ru.yandex.tours.direction.layout.Layout
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsVideo
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.Prices._
import ru.yandex.tours.model._
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.purchase.{BillingObject, PurchaseObject, PurchaseWithGet, PurchaseWithPost}
import ru.yandex.tours.model.search.SearchProducts.{Actualization, Offer}
import ru.yandex.tours.model.search.SearchResults.{ActualizedOffer, ErrorCode, ResultInfo, SearchProgress}
import ru.yandex.tours.model.search.{HotelSearchRequest, SearchType, WhereToGoRequest}
import ru.yandex.tours.model.util.proto._
import ru.yandex.tours.model.util.{Paging, SortType, proto}
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.serialize.json.CommonJsonSerialization
import ru.yandex.tours.util.http.{UrlLabelerBuilder, UrlQueryPatcher}

import scala.collection.JavaConversions._

/* @author berkut@yandex-team.ru */

trait JsonSerialization extends CommonJsonSerialization {

  def toJson(array: JSONArray, paging: Paging, totalSize: Int): JSONObject = {
    new JSONObject()
      .put("list", array)
      .put("pager", new JSONObject()
        .put("count", totalSize)
        .put("current", paging.page + 1)
        .put("page_size", paging.pageSize)
        .put("available_page_count", (totalSize / paging.pageSize) + (if (totalSize % paging.pageSize == 0) 0 else 1)))
  }

  def toJson(progress: SearchProgress): JSONObject = {
    new JSONObject()
      .put("done", progress.getOperatorCompleteCount)
      .put("total", progress.getOperatorTotalCount)
      .put("is_complete", progress.getIsFinished)
  }

  def toJson(progress: SearchProgress, found: Int, total: Int): JSONObject = {
    toJson(progress)
      .put("items", found)
      .put("total_items", total)
  }

  def toJson(offerStatistic: OfferStatistic): JSONObject = {
    val res = new JSONObject()
    offerStatistic.minPrice.foreach(price => res.put("min_price", price))
    res
  }

  def toJson(graph: PriceGraph, tree: Tree, lang: Lang): JSONObject = {
    val prices = new JSONArray()
    graph.getPriceEntityList.foreach(e => prices.put(e.getMinPrice))
    new JSONObject()
      .put("index", graph.getIndex)
      .put("nights", graph.getNights)
      .put("ages", agesToArray(graph.getAgeList.map(Int.unbox)))
      .put("prices", prices)
      .put("to", regionToJson(graph.getTo, tree, lang))
      .put("when", ru.yandex.tours.api.formatter.print(toLocalDate(graph.getWhen)))
  }

  def toJson(bestPrice: DirectionBestPrice, urlBuilder: UrlBuilder, utm: UtmMark): JSONObject = {
    val request = bestPrice.getSearchRequest
    val when = proto.toLocalDate(request.getWhen)
    val url = urlBuilder.forSearch(request.getFrom, request.getTo, when, request.getNights,
      utm = utm, context = SearchType.TOURS)
    new JSONObject()
      .put("to", bestPrice.getSearchRequest.getTo)
      .put("when", when.toString)
      .put("min_price", bestPrice.getPrice)
      .put("nights", request.getNights)
      .put("url", url)
  }

  def toJson[T <: Source](offer: Offer,
                          lang: Lang,
                          operators: SearchSources[T],
                          billingObject: Option[BillingObject],
                          labeler: UrlQueryPatcher): JSONObject = {
    val fly = (offer.getWithTransfer, offer.getWithFlight) match {
      case (true, true) => "trans_and_fly"
      case (false, false) => "no_fly_no_trans"
      case (true, false) => "trans_no_fly"
      case (false, true) => "fly_no_trans"
    }
    val cancellation_type = (offer.hasFreeCancellation, offer.getFreeCancellation) match {
      case (false, _) => "unknown"
      case (true, true) => "with_free_cancellation"
      case (true, false) => "without_free_cancellation"
    }
    val ans = new JSONObject()
      .put("id", offer.getId)
      .put("start_date", formatter.print(toLocalDate(offer.getDate)))
      .put("nights", offer.getNights)
      .put("pansion", offer.getPansion)
      .put("room_type", offer.getRoomType)
      .put("price", offer.getPrice)
      .put("fly", fly)
      .put("with_transfer", offer.getWithTransfer)
      .put("with_flight", offer.getWithFlight)
      .put("with_medical_insurance", offer.getWithMedicalInsurance)
      .put("conditions", new JSONObject()
        .put("with_free_cancellation", offer.hasFreeCancellation && offer.getFreeCancellation) // Deprecated field
        .put("cancellation_type", cancellation_type)
      )
    if (offer.hasAgentBookingUrl) ans.put("link", labeler(offer.getAgentBookingUrl))
    billingObject.foreach(addBillingInfo(ans, _))
    operators.getById(offer.getSource.getOperatorId).foreach(operator => {
      ans
        .put("operator_name", operator.name)
        .put("operator_id", operator.id)
    })
    if (operators.getById(offer.getSource.getOperatorId).exists(_.code == "view_on_site"))
      ans.put("is_cmhotel", true)
    ans
  }

  def toJson(actualizedOffer: ActualizedOffer, purchases: Seq[PurchaseObject], tree: Tree, lang: Lang, labeler: UrlQueryPatcher): JSONObject = {
    val ans = new JSONObject()
      .put("links", linksToJson(purchases, labeler))
      .put("price", actualizedOffer.getOffer.getPrice)
    if (actualizedOffer.hasActualizationInfo) {
      val ai = actualizedOffer.getActualizationInfo
      ans
        .put("fuel_charge", ai.getFuelCharge)
        .put("price", ai.getPrice)
      if (ai.getFlightsCount > 0) ans.put("flight", toJson(ai.getFlightsList, tree, lang))
      if (ai.hasInfantPrice) ans.put("infant_price", ai.getInfantPrice)
      if (ai.hasWithTransfer) ans.put("with_transfer", ai.getWithTransfer)
    }
    ans
  }

  def toJson(flight: Iterable[Actualization.Flight], tree: Tree, lang: Lang): JSONObject = {
    val departure = flight.map(_.getTo)
    val arrival = flight.map(_.getBack)
    new JSONObject()
      .put("departure", routesToJson(departure, tree, lang))
      .put("arrival", routesToJson(arrival, tree, lang))
  }

  def toJson(resultInfo: ResultInfo, tooCloseDestination: Boolean): JSONObject = {
    val ans = new JSONObject()
    ans.put("from_long_cache", resultInfo.getIsFromLongCache)
    if (resultInfo.hasError) {
      ans.put("error_code", resultInfo.getError.getNumber)
      ans.put("error_name", resultInfo.getError.name)
    }
    if ((resultInfo.hasError && resultInfo.getError == ErrorCode.TOO_CLOSE_DESTINATION) || tooCloseDestination) {
      ans.put("too_close_destination", true)
    }
    ans
  }

  def toJson(resultInfo: ResultInfo, countrySearch: Boolean,
             filtersEnabled: Boolean, allFiltered: Boolean): JSONObject = {
    val ans = toJson(resultInfo, tooCloseDestination = false)
    if (allFiltered || resultInfo.hasError) {
      val message = if (resultInfo.hasError) resultInfo.getError else "ALL_FILTERED"
      ans.put("empty_reason", message)
    }
    ans.put("country_search", countrySearch)
    ans.put("filters_enabled", filtersEnabled)
    ans
  }

  private def routesToJson(routes: Iterable[Actualization.Route], tree: Tree, lang: Lang): JSONObject = {
    require(routes.nonEmpty, "routes should not be empty")
    val jsonRoutes = new JSONArray()
    for (route <- routes.toSeq.distinct.sortBy(_.getPoint(0).getDeparture)) {
      jsonRoutes.put(
        new JSONObject()
          .put("route", toJson(route, tree, lang))
          .put("is_direct", route.getIsDirect)
      )
    }
    val from = routes.head.getPointList.head.getGeoId
    val to = routes.head.getPointList.last.getGeoId
    val when = routes.head.getPointList.head.getDeparture
    new JSONObject()
      .put("from", regionName(from, tree, lang))
      .put("to", regionName(to, tree, lang))
      .put("when", when)
      .put("routes", jsonRoutes)
  }

  def toJson(x: Actualization.Route, tree: Tree, lang: Lang): JSONArray = {
    val route = new JSONArray()
    val routeSize = x.getPointCount
    x.getPointList.zipWithIndex.foreach({ case (point, index) =>
      val t = toJson(point, tree, lang)
      val `type` = index match {
        case 0 => "from"
        case z if z == routeSize - 1 => "to"
        case _ => "change"
      }
      t.put("type", `type`)
      route.put(t)
    })
    route
  }

  def toJson(flightPoint: Actualization.FlightPoint, tree: Tree, lang: Lang): JSONObject = {
    val ans = new JSONObject()
      .put("name", regionName(flightPoint.getGeoId, tree, lang))
      .put("company", flightPoint.getCompany)
    if (flightPoint.hasAirport) {
      ans.put("airport", flightPoint.getAirport)
    }
    if (flightPoint.hasDeparture && !flightPoint.getDeparture.endsWith("00:00:00.000Z")) {
      ans.put("departure_time", flightPoint.getDeparture)
    }
    if (flightPoint.hasArrival && !flightPoint.getArrival.endsWith("00:00:00.000Z")) {
      ans.put("arrival_time", flightPoint.getArrival)
    }
    ans
  }

  def toJson(direction: RecommendedDirection, geoId: Int, lang: Lang, tree: Tree): JSONObject = {
    toJson(direction, geoId, lang, tree, squareLayout = false)
  }

  def toJson(direction: RecommendedDirection, geoId: Int, lang: Lang, tree: Tree, squareLayout: Boolean): JSONObject = {
    val request = HotelSearchRequest(direction.getSearchRequest)
    val ans = new JSONObject()
      .put("search_request", toJson(request, tree, lang, sortBy = Some(SortType.PRICE_ASC)))
      .put("name", regionName(direction.getGeoId, tree, lang))
      .put("image", toJson(Image.fromProto(direction.getImage)))
      .put("ages", new JSONArray(DefaultRequestGenerator.DEFAULT_AGES.toArray))
    if (request.from != geoId && request.from != 0) {
      ans.put("from", regionToJson(request.from, tree, lang))
    }
    for (country <- tree.country(direction.getGeoId)) {
      ans.put("country", regionName(country.id, tree, lang))
    }
    if (squareLayout && direction.hasSquareImage) {
      ans.put("image", toJson(Image.fromProto(direction.getSquareImage)))
    }
    if (direction.hasPrice) {
      ans.put("price_from", direction.getPrice)
    }
    if (direction.getHasCard) {
      ans.put("card", direction.getGeoId)
    }
    if (direction.hasBrandingCampaign) {
      ans.put("branding_campaign", direction.getBrandingCampaign)
    }
    def bestPriceToJson(bestPrice: DirectionBestPrice): JSONObject = {
      val request = HotelSearchRequest(bestPrice.getSearchRequest)
      val res = new JSONObject()
        .put("search_request", toJson(request, tree, lang, sortBy = Some(SortType.PRICE_ASC)))
        .put("price_from", bestPrice.getPrice)
      if (bestPrice.hasWithFlight && bestPrice.hasFlightPrice) {
        res
          .put("with_flight", bestPrice.getWithFlight)
          .put("flight_price", bestPrice.getFlightPrice)
      }
      if (bestPrice.hasRoomPrice) {
        res.put("room_price", bestPrice.getRoomPrice)
      }
      res
    }

    if (direction.hasTourPrice) {
      ans
        .put("tours", bestPriceToJson(direction.getTourPrice))
        .put("nights", direction.getTourPrice.getSearchRequest.getNights)
    }
    if (direction.hasRoomPrice) {
      ans
        .put("rooms", bestPriceToJson(direction.getRoomPrice))
        .put("nights", direction.getRoomPrice.getSearchRequest.getNights)
    }
    ans
  }

  def addBillingInfo(parent: JSONObject, billingObject: BillingObject) = {
    val billingInfo = new JSONObject()
    billingObject.uiInfo.foreach({ case (key, value) => billingInfo.put(key, value) })
    parent
      .put("billing", billingInfo)
      .put("offer_campaign_id", billingObject.campaignId)
  }

  def linksToJson(links: Seq[PurchaseObject], labeler: UrlQueryPatcher): JSONArray = {
    val ans = new JSONArray()
    links.foreach(purchaseObject => {
      val purchase = new JSONObject()
        .put("billing_id", purchaseObject.billingId)
        .put("name", purchaseObject.partnerName)
      purchaseObject match {
        case PurchaseWithGet(link, _, _, _, isTourOperator) =>
          purchase.put("link", labeler(link))
          purchase.put("is_tour_operator", isTourOperator)
        case PurchaseWithPost(link, offer, signature, _, _, _) =>
          purchase.put("link", link)
          purchase.put("post_params", new JSONObject()
            .put("offer", offer)
            .put("hash", signature)
            .put(UrlLabelerBuilder.defaultLabelName, labeler.queries.getOrElse(UrlLabelerBuilder.defaultLabelName, ""))
          )
      }
      addBillingInfo(purchase, purchaseObject.billing)
      ans.put(purchase)
    })
    ans
  }


  def agesToArray(ages: Iterable[Int]): JSONArray = {
    val jsonAges = new JSONArray()
    ages.foreach(jsonAges.put)
    jsonAges
  }

  def snippetToJson[T <: Source](snippetWithBilling: SnippetWithBilling,
                                 ages: Iterable[Int],
                                 tree: Tree,
                                 searchSources: SearchSources[T],
                                 hotelsVideo: HotelsVideo,
                                 geoMappingHolder: GeoMappingHolder,
                                 searchSettingsHolder: SearchSettingsHolder,
                                 lang: Lang,
                                 labeler: UrlQueryPatcher): JSONObject = {
    val result = new JSONObject()
    val tourInfo = snippetWithBilling.snippet.snippet
    val hotel = snippetWithBilling.snippet.hotel
    val hotelJson = toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang)

    snippetWithBilling.hotelBilling.foreach { idedBilling =>
      val billing = new JSONObject()
      BillingOffers.dumpOffer(idedBilling).toOption.foreach { offer =>
        offer.foreach({ case (key, value) => billing.put(key, value) })
      }
      hotelJson.put("billing", billing)
      hotelJson.put("is_paid", true)
    }
    result.put("hotel", hotelJson)
    result.put("price_from", tourInfo.getPriceMin)
    result.put("price_to", tourInfo.getPriceMax)
    result.put("start_date_from", formatter.print(toLocalDate(tourInfo.getDateMin)))
    result.put("start_date_to", formatter.print(toLocalDate(tourInfo.getDateMax)))
    result.put("nights_from", tourInfo.getNightsMin)
    result.put("nights_to", tourInfo.getNightsMax)
    result.put("nights", tourInfo.getNightsMin)
    if (tourInfo.hasLandingPage) {
      val landing = new JSONObject().put("url", labeler(tourInfo.getLandingPage.getUrl))
      searchSources.getById(tourInfo.getLandingPage.getProviderId).foreach(provider => {
        landing.put("provider_name", provider.name)
        landing.put("provider_id", provider.id)
      })
      result.put("landing_page", landing)
    }
    val pansions = new JSONArray()
    tourInfo.getPansionsList.map(pp => {
      pansions.put(pp.getPansion.toString)
    })
    result.put("pansions", pansions)
    result.put("ages", agesToArray(ages))
    result.put("offer_count", tourInfo.getOfferCount)
    if (tourInfo.hasSample) {
      result.put("offer", toJson(tourInfo.getSample, lang, searchSources, snippetWithBilling.offerBilling, labeler))
    }
    val operators = tourInfo.getSourceList.map(_.getOperatorId).distinct.flatMap(searchSources.getById)
    result.put("tour_operators", operators.foldLeft(new JSONArray()) {
      case (arr, operator) =>
        val obj = new JSONObject()
          .put("operator_name", operator.name)
          .put("operator_id", operator.id)
        arr.put(obj)
    })
    result
  }

  def mapHotelToJson(hotels: Iterable[Hotel], minPrices: Map[Hotel, HotelMinPrice], lang: Lang): JSONArray = {
    val ans = new JSONArray()
    for (hotel <- hotels) {
      val obj = new JSONObject()
        .put("id", hotel.id)
        .put("stars", hotel.star.id)
        .put("name", hotel.name(lang))
        .put("latitude", hotel.latitude)
        .put("longitude", hotel.longitude)
        .put("rating", hotel.rating)
        .put("reviews_count", hotel.reviewsCount)
      minPrices.get(hotel).foreach { minPrice =>
        obj.put("min_prices", minPriceToJson(minPrice))
      }
      ans.put(obj)
    }
    ans
  }

  def minPriceToJson(hotelMinPrice: HotelMinPrice): JSONObject = {
    val ans = new JSONObject()
    ans.put("is_complete", hotelMinPrice.isComplete)
    hotelMinPrice.tourMinPrice.foreach { ans.put("tour_price", _) }
    hotelMinPrice.roomMinPrice.foreach { ans.put("room_price", _) }
    ans
  }

  def toJson(request: WhereToGoRequest): JSONObject = {
    val ans = new JSONObject()
    ans
      .put("thematic", request.thematic.toString.toLowerCase)
      .put("no_visa", request.noVisa.toString)

    request.budget.foreach(ans.put("budget", _))
    request.month.foreach(m => ans.put("month", m.getValue.toString))
    request.countryId.foreach(c => ans.put("country", c.toString))
    ans
  }

  def toJson(layout: Layout, directions: Seq[RecommendedDirection], geoId: Int, tree: Tree, lang: Lang): JSONArray = {
    require(layout.size == directions.size, "Layout size != directions.size")
    val result = new JSONArray()
    val it = directions.iterator
    for (row <- layout.rows) {
      val rowArray = new JSONArray()
      for (item <- row.items) {
        val isSquare = (item.size == 1) || (item.size == 4)
        val direction = it.next()
        val json = new JSONObject()
        json
          .put("data", toJson(direction, geoId, lang, tree, isSquare))
          .put("position", item.position)
          .put("size", item.size)

        rowArray.put(json)
      }
      result.put(rowArray)
    }
    result
  }
}

object JsonSerialization extends JsonSerialization