package ru.yandex.tours.api

import org.joda.time.{DateTime, DateTimeZone}
import org.slf4j.LoggerFactory
import ru.yandex.tours.backend.HotelSnippetPreparer.{HotelsWithInfo, OffersWithInfo}
import ru.yandex.tours.model.filter.Filter
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.SearchResults.{ErrorCode, FlightSearchResult}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{FlightSearchRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.ReqAnsFormat

import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.04.15
 */
class ReqAnsLogger(format: String) {
  private val log = LoggerFactory.getLogger("tours.api.stats.logs")

  private val project = "tours"
  private val locale = "ru"
  private val formatVersion = "1"
  private val component = "api_reqans"

  private val prefix = s"tskv\ttskv_format=$format\tformat_version=$formatVersion\tlocale=$locale\tproject=$project"

  private def logTSKV(data: Map[String, Any]) = {
    val ts = DateTime.now(DateTimeZone.UTC)
    val sb = new mutable.StringBuilder(prefix)
    sb.append("\tcomponent=").append(component)
    sb.append("\ttimestamp=").append(ts)
    sb.append("\tunixtime=").append(ts.getMillis / 1000)
    for ((key, value) <- data) {
      sb ++= "\t"
      sb ++= key
      sb ++= "="
      sb ++= value.toString
    }
    log.info(sb.toString())
  }

  def logHotels(reqIndex: Int,
                user: UserIdentifiers,
                req: HotelSearchRequest,
                page: Paging,
                filters: Seq[Filter],
                sortBy: SortType,
                result: HotelsWithInfo,
                context: SearchType): Unit = {
    val data = req.asMap ++ Seq(
      "type" -> "hotels_search",
      "req_id" -> req.sessionId,
      "filters" -> JsonSerialization.filtersToJson(filters.filter(_.nonEmpty)),
      "page" -> page.page,
      "page_size" -> page.pageSize,
      "sort" -> sortBy.toString,
      "is_finished" -> result.progress.getIsFinished,
      "serp_id" -> ReqAnsFormat.serpId(DateTime.now, user, context, req),
      "yuid" -> user.yuid.getOrElse(""),
      "uid" -> user.uid.getOrElse(""),
      "login" -> user.login.getOrElse(""),
      "hotels_shown" -> result.statistic.found,
      "hotels_total" -> result.statistic.total,
      "tours_shown" -> result.statistic.offerStatistic.found,
      "tours_total" -> result.statistic.offerStatistic.total,
      "req_index" -> reqIndex,
      "context" -> context,
      "too_large_dest" -> (result.resultInfo.hasError && result.resultInfo.getError == ErrorCode.TOO_LARGE_DESTINATION),
      "snippets" -> result.snippets.map(_.snippet.snippet).map(ReqAnsFormat.snippet).mkString("[", ",", "]")
    )
    logTSKV(data)
  }

  def logOffers(reqIndex: Int,
                req: OfferSearchRequest,
                page: Paging,
                filters: Seq[Filter],
                sortBy: SortType,
                result: OffersWithInfo,
                user: UserIdentifiers,
                context: SearchType): Unit = {
    val data = req.asMap ++ Seq(
      "type" -> "offers_search",
      "req_id" -> req.requestId,
      "filters" -> JsonSerialization.filtersToJson(filters.filter(_.nonEmpty)),
      "page" -> page.page,
      "page_size" -> page.pageSize,
      "sort" -> sortBy.toString,
      "is_finished" -> result.progress.getIsFinished,
      "yuid" -> user.yuid.getOrElse(""),
      "uid" -> user.uid.getOrElse(""),
      "login" -> user.login.getOrElse(""),
      "tours_shown" -> result.statistic.found,
      "tours_total" -> result.statistic.total,
      "req_index" -> reqIndex,
      "context" -> context,
      "offers" -> result.tours.map(_.offer).map(ReqAnsFormat.offer).mkString("[", ",", "]")
    )
    logTSKV(data)
  }

  def logTourOffer(req: OfferSearchRequest, offer: Offer, context: SearchType, user: UserIdentifiers): Unit = {
    val data = req.asMap ++ Map(
      "type" -> "tour_show",
      "offer_id" -> offer.getId,
      "yuid" -> user.yuid.getOrElse(""),
      "uid" -> user.uid.getOrElse(""),
      "login" -> user.login.getOrElse(""),
      "context" -> context,
      "offer" -> ReqAnsFormat.offer(offer)
    )
    logTSKV(data)
  }

  def logFlights(reqIndex: Int,
                 req: FlightSearchRequest,
                 result: FlightSearchResult,
                 user: UserIdentifiers): Unit = {
    val data = req.asMap ++ Seq(
      "type" -> "flights_show",
      "req_id" -> req.sessionId,
      "is_finished" -> result.getProgress.getIsFinished,
      "yuid" -> user.yuid.getOrElse(""),
      "uid" -> user.uid.getOrElse(""),
      "login" -> user.login.getOrElse(""),
      "req_index" -> reqIndex
    )
    logTSKV(data)
  }
}
