package ru.yandex.tours.api.v1.hotel

import javax.ws.rs.Path

import com.wordnik.swagger.annotations._
import spray.routing.HttpService

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 01.02.16
 */
@Api(
  basePath = "/hotel/{id}",
  value = "/hotel",
  description = "Все про конкретный отель",
  produces = "application/json",
  position = 0
)
trait Documentation extends HttpService {

  //scalastyle:off

  @Path("/info")
  @ApiOperation(httpMethod = "GET", value = "Информация про отель", position = 0, notes =
    """ID отеля в ответе может не совпадать с ID переданным в запросе,
       в этом случае вернется статус 308 с редиректом на новый отель.
    """)
  @ApiImplicitParams(Array(
    new ApiImplicitParam(name = "id", dataType = "int", required = true, value = "id отеля", paramType = "path")
  ))
  @ApiResponses(Array(
    new ApiResponse(code = 200, message = "JSON с информацией про отель"),
    new ApiResponse(code = 308, message = "JSON с информацией про отель и редирект на актуальный ID"),
    new ApiResponse(code = 404, message = "Такого отеля больше нет")
  ))
  def info()

  @Path("/{context}/search")
  @ApiOperation(httpMethod = "GET", value = "Поиск офферов в отель", position = 1, notes =
    """Необходимо вызывать до тех пор пока поле progress.is_complete не примет значение true.
       С каждым новым запросом необходимо увеличивать значение параметра request_index (начиная с 0).

       Параметр to не обязателен, но рекомендуется передать такой же, как при поиске по региону.
       Такое поведение позволит получить первые результаты немного быстрее.

       При поиске с context = rooms, параметр from не обязателен.
       Параметры flex_when и flex_nights работают только с context = tours.

       Если параметры when или nights не переданы, сервис попытается найти ближайшие к текущей дате, когда есть предложения.

       Рекомендуется передавать utm метки пользователя, а так же его идентификатор.
       Идентификатор пользователя можно передать параметрами uid, yuid и login или заголовком X-UserId в формате uid/login/yuid, например, X-UserId: 123/my_login/234414455

       Допустимые фильтры:
         * pansions – ... https://hg.yandex-team.ru/tours/file/tip/tours-model-proto/src/main/proto/base_model.proto#l3
         * price_from, price_to
         * tour_operator | hotel_provider
         * exclude_id – игнорирование конкретного тура
    """)
  @ApiImplicitParams(Array(
    new ApiImplicitParam(name = "id", dataType = "int", required = true, value = "id отеля", paramType = "path"),
    new ApiImplicitParam(name = "context", dataType = "string", paramType = "path", allowableValues = "tours,rooms", required = true, defaultValue = "tours"),
    new ApiImplicitParam(name = "from", dataType = "int", paramType = "query", value = "id города вылета"),
    new ApiImplicitParam(name = "to", dataType = "int", paramType = "query", value = "id города прилета (не используется)"),
    new ApiImplicitParam(name = "when", dataType = "date", paramType = "query"),
    new ApiImplicitParam(name = "nights", dataType = "int", paramType = "query"),
    new ApiImplicitParam(name = "ages", dataType = "int", paramType = "query", allowMultiple = true),
    new ApiImplicitParam(name = "flex_when", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 дня к дате вылета"),
    new ApiImplicitParam(name = "flex_nights", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 к числу ночей"),
    new ApiImplicitParam(name = "lang", dataType = "string", paramType = "query", allowableValues = "ru,en,kz,ua", defaultValue = "ru"),
    new ApiImplicitParam(name = "request_index", dataType = "int", paramType = "query", defaultValue = "0"),
    new ApiImplicitParam(name = "page_num", dataType = "int", paramType = "query", defaultValue = "1"),
    new ApiImplicitParam(name = "page_size", dataType = "int", paramType = "query", defaultValue = "20"),
    new ApiImplicitParam(name = "sort", dataType = "string", paramType = "query", defaultValue = "relevance", allowableValues = "relevance,price_asc,price_desc")
  ))
  def search()

  @Path("/min_prices")
  @ApiOperation(httpMethod = "GET", value = "Поиск минимальных цен на отель", position = 2, notes =
    """Поиск минимальных цен на тур и комнату в конкретный отель.

       Необходимо вызывать до тех пор пока поле is_complete не примет значение true.

       Параметр to не обязателен, но рекомендуется передать такой же, как при поиске по региону.
       Такое поведение позволит получить первые результаты немного быстрее.

       Если параметры when или nights не переданы, сервис попытается найти ближайшие к текущей дате, когда есть предложения.
       Но есть опасность, что поиск никогда не завершится.
    """)
  @ApiImplicitParams(Array(
    new ApiImplicitParam(name = "id", dataType = "int", required = true, value = "id отеля", paramType = "path"),
    new ApiImplicitParam(name = "from", dataType = "int", paramType = "query", value = "id города вылета"),
    new ApiImplicitParam(name = "to", dataType = "int", paramType = "query", value = "id города прилета (не используется)"),
    new ApiImplicitParam(name = "when", dataType = "date", paramType = "query"),
    new ApiImplicitParam(name = "nights", dataType = "int", paramType = "query"),
    new ApiImplicitParam(name = "ages", dataType = "int", paramType = "query", allowMultiple = true),
    new ApiImplicitParam(name = "flex_when", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 дня к дате вылета"),
    new ApiImplicitParam(name = "flex_nights", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 к числу ночей")
  ))
  def minPrices()

  @Path("/tour/{tour_id}")
  @ApiOperation(httpMethod = "GET", value = "Информация про тур", position = 3, notes =
    """Параметры поиска должны совпадать с параметрами при которых получен тур.

       Рекомендуется передавать utm метки пользователя, а так же его идентификатор.
       Идентификатор пользователя можно передать параметрами uid, yuid и login или заголовком X-UserId в формате uid/login/yuid, например, X-UserId: 123/my_login/234414455

       ota_count влияет на максимальное число кнопок для онлайн покупки.
    """)
  @ApiImplicitParams(Array(
    new ApiImplicitParam(name = "id", dataType = "int", required = true, value = "id отеля", paramType = "path"),
    new ApiImplicitParam(name = "tour_id", dataType = "string", paramType = "path", required = true),
    new ApiImplicitParam(name = "from", dataType = "int", paramType = "query", value = "id города вылета"),
    new ApiImplicitParam(name = "to", dataType = "int", paramType = "query", value = "id города прилета (не используется)"),
    new ApiImplicitParam(name = "when", dataType = "date", paramType = "query"),
    new ApiImplicitParam(name = "nights", dataType = "int", paramType = "query"),
    new ApiImplicitParam(name = "ages", dataType = "int", paramType = "query", allowMultiple = true),
    new ApiImplicitParam(name = "flex_when", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 дня к дате вылета"),
    new ApiImplicitParam(name = "flex_nights", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 к числу ночей"),
    new ApiImplicitParam(name = "lang", dataType = "string", paramType = "query", allowableValues = "ru,en,kz,ua", defaultValue = "ru"),
    new ApiImplicitParam(name = "ota_count", dataType = "int", paramType = "query", defaultValue = "3")
  ))
  def tour()

  @Path("/tour/{tour_id}/actualization")
  @ApiOperation(httpMethod = "GET", value = "Актуализация информации про тур", position = 4, notes =
    """Параметры поиска должны совпадать с параметрами при которых получен тур.

       Актуализация обновляет цену на тур, узнает цену топливного сбора, сборы за инфантов, наличие трансфера и мед. страховки,
       а так же информацию про возможные авиа перелеты.

       Опирация синхронная, и достаточно тяжелая для партнеров.
       Может выполняться до 60 секунд.

       Рекомендуется передавать utm метки пользователя, а так же его идентификатор.
       Идентификатор пользователя можно передать параметрами uid, yuid и login или заголовком X-UserId в формате uid/login/yuid, например, X-UserId: 123/my_login/234414455

       ota_count влияет на максимальное число кнопок для онлайн покупки.
       После актуализации данные для кнопок могут измениться.
    """)
  @ApiImplicitParams(Array(
    new ApiImplicitParam(name = "id", dataType = "int", required = true, value = "id отеля", paramType = "path"),
    new ApiImplicitParam(name = "tour_id", dataType = "string", paramType = "path", required = true),
    new ApiImplicitParam(name = "from", dataType = "int", paramType = "query", value = "id города вылета"),
    new ApiImplicitParam(name = "to", dataType = "int", paramType = "query", value = "id города прилета (не используется)"),
    new ApiImplicitParam(name = "when", dataType = "date", paramType = "query"),
    new ApiImplicitParam(name = "nights", dataType = "int", paramType = "query"),
    new ApiImplicitParam(name = "ages", dataType = "int", paramType = "query", allowMultiple = true),
    new ApiImplicitParam(name = "flex_when", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 дня к дате вылета"),
    new ApiImplicitParam(name = "flex_nights", dataType = "boolean", paramType = "query", defaultValue = "false", value = "+- 2 к числу ночей"),
    new ApiImplicitParam(name = "lang", dataType = "string", paramType = "query", allowableValues = "ru,en,kz,ua", defaultValue = "ru"),
    new ApiImplicitParam(name = "ota_count", dataType = "int", paramType = "query", defaultValue = "3")
  ))
  def tourActualization()
}
