package ru.yandex.tours.api.v1.hotel

import org.json.JSONObject
import ru.yandex.tours.api.ReqAnsLogger
import ru.yandex.tours.backend.{OfferPreparer, PurchaseService}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsVideo
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{GetOfferRequest, OfferSearchRequest}
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.http.UrlLabelerBuilder
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.LabelBuilder
import spray.http.StatusCodes
import spray.routing.Directives._
import spray.routing.Route

import scala.concurrent.ExecutionContext

/**
 * Tour offer related routes:
 * {{{
 * /tour/<tour-id>?ota_count=3&<utm>&<user>
 * /tour/<tour-id>/actualization?<request>&<lang>&ota_count=3&<utm>&<user>
 * }}}
 */
class GetOfferRouter(routeesContext: RouteesContext,
                     offerPreparer: OfferPreparer,
                     purchaseService: PurchaseService,
                     geoMapping: GeoMappingHolder,
                     searchSettings: SearchSettingsHolder,
                     tree: Tree,
                     tourOperators: TourOperators,
                     searchType: SearchType,
                     hotelsVideo: HotelsVideo,
                     reqAnsLogger: ReqAnsLogger,
                     redirectUrl: String,
                     labelBuilder: LabelBuilder)
                    (implicit ec: ExecutionContext)
  extends OfferSearchRouter(routeesContext, offerPreparer, tree, tourOperators, searchType,
    geoMapping, searchSettings, hotelsVideo, reqAnsLogger, redirectUrl, labelBuilder)(ec) {

  override def route(hotel: Hotel): Route = super.route(hotel) ~ routes(hotel)

  private def routes(hotel: Hotel): Route = {
    (pathPrefix("tour" / Segment) & routeesContext.offersSearchRequest(hotel) & utm & userIdentifiers &
      lang & parameter('ota_count.as[Int] ? 3)) {
      (offerId, request, utm, userIdentifiers, lang, visibleButtonCount) =>
        getOffer(hotel, request, offerId, utm, userIdentifiers, lang, visibleButtonCount) ~
          actualizeOffer(hotel, request, offerId, utm, userIdentifiers, lang, visibleButtonCount)
    }
  }

  private def getOffer(hotel: Hotel, request: OfferSearchRequest, offerId: String, utm: UtmMark,
                       userIdentifiers: UserIdentifiers, lang: Lang, visibleButtonCount: Int): Route = {
    (pathEndOrSingleSlash & metered("tour_info")) {
      val getOfferRequest = GetOfferRequest(request, offerId)
      onSuccess(offerPreparer.getOffer(getOfferRequest)) {
        case Some(actualizedOffer) =>
          val purchases = purchaseService.getPurchaseLinks(request, actualizedOffer,
            utm, userIdentifiers, visibleButtonCount)
          val labeler = UrlLabelerBuilder(utm, userIdentifiers, redirectUrl, labelBuilder).buildFrom(actualizedOffer)
          val ans = new JSONObject()
            .put("hotel", toJson(hotel, hotelsVideo, tree, geoMapping, searchSettings, lang))
            .put("search_request", toJson(getOfferRequest, tree, lang))
            .put("tour", toJson(actualizedOffer.getOffer, lang, tourOperators, None, labeler))
            .put("links", linksToJson(purchases, labeler))
          reqAnsLogger.logTourOffer(request, actualizedOffer.getOffer, searchType, userIdentifiers)
          completeJsonOk(ans)
        case None =>
          completeJsonError(StatusCodes.NotFound, "Unknown tour id")
      }
    }
  }

  private def actualizeOffer(hotel: Hotel, request: OfferSearchRequest, offerId: String, utm: UtmMark,
                             userIdentifiers: UserIdentifiers, lang: Lang, visibleButtonCount: Int): Route = {
    (path("actualization") & metered("tour_actualization")) {
      val tourRequest = GetOfferRequest(request, offerId)
      onSuccess(offerPreparer.getActualizedOffer(tourRequest)) {
        case Some(actualizedOffer) =>
          val purchases = purchaseService.getPurchaseLinks(request, actualizedOffer,
            utm, userIdentifiers, visibleButtonCount)

          val labeler = UrlLabelerBuilder(utm, userIdentifiers, redirectUrl, labelBuilder).buildFrom(actualizedOffer)
          completeJsonOk(toJson(actualizedOffer, purchases, tree, lang, labeler))
        case None =>
          completeJsonError(StatusCodes.NotFound, "Unknown tour id")
      }
    }
  }
}
