package ru.yandex.tours.api.v1.hotel

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.avia.AviaAirportRecommendations
import ru.yandex.tours.backend.{HotelMinPricePreparer, HotelSnippetPreparer}
import ru.yandex.tours.direction.DirectionsStats
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsIndex, HotelsPanoramas, HotelsVideo}
import ru.yandex.tours.model.{HotelProvider, TourOperator}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.resorts.SkiResorts
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing.Route

/**
 * Hotel related routes:
 * {{{
 * /<hotel-id>/info?<lang>&from
 * /<hotel-id>/near_hotels?<map-rectangle>&max_size=200
 * /<hotel-id>/near_hotels?max_size=10&<lang>
 * /<hotel-id>/near_cities?max_size=5&<lang>
 * /<hotel-id>/near_countries?max_size=5&<lang>
 * /<hotel-id>/<context>/ -> OfferSearchRouter
 * /<hotel-id>/similar?<request>&max_size=3&<lang>
 * /<hotel-id>/search?<request>&<lang>&<paging>&<sorting>&<filters>
 * /<hotel-id>/tour/<tour-id>?<request>&<lang>&<utm>&<user>&<ota_count>
 * /<hotel-id>/tour/<tour-id>/actualization?<request>&<lang>&<utm>&<user>&<ota_count>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class HotelHandler(routeesContext: RouteesContext,
                   hotelIndex: HotelsIndex,
                   tree: Tree,
                   geoMapping: GeoMappingHolder,
                   searchSettings: SearchSettingsHolder,
                   directionsStats: DirectionsStats,
                   toursOfferSearchRoute: Hotel => Route,
                   roomsOfferSearchRoute: Hotel => Route,
                   hotelMinPricePreparer: HotelMinPricePreparer,
                   hotelsVideo: HotelsVideo,
                   hotelsPanoramas: HotelsPanoramas,
                   aviaAirportRecommendations: AviaAirportRecommendations,
                   skiResorts: SkiResorts) extends HttpHandler with JsonSerialization with Bindings {

  override protected def metered(name: String) = super.metered("hotel." + name)

  override def route: Route = (pathPrefix(IntNumber) & lang & metered("root")) { (hotelId, lang) =>
    withHotel(hotelIndex, hotelId) { hotel ⇒
      (path("info") & metered("info")) {
        val hotelJson = toJson(hotel, hotelsVideo, tree, geoMapping, searchSettings, lang)
        toJson(hotel, hotelsPanoramas).foreach(hotelJson.put("panorama", _))

        val ans = new JSONObject().put("hotel", hotelJson)

        tree.pathToRoot(hotel.geoId).find(r => skiResorts.get(r.id).isDefined).foreach { r =>
          ans.put("resort_id", r.id)
        }
        if (hotelId != hotel.id) {
          val uri = Uri(s"../${hotel.id}/info").withQuery("lang" -> lang.toString)
          completeJsonRedirect(uri, ans)
        } else {
          completeJsonOk(ans)
        }
      } ~ (path("min_prices") & routeesContext.offersSearchRequest(hotel) & metered("min_prices")) { request =>
        onSuccess(hotelMinPricePreparer.getHotelMinPrice(request.hotelRequest, hotel)) {
          minPrice => completeJsonOk(minPriceToJson(minPrice))
        }
      } ~ path("near_hotels") {
        (mapRectangle & parameter('max_size.as[Int] ? 200) & metered("near_hotels_with_rect")) { (mapInfo, maxSize) =>
          val nearHotels = hotelIndex.inRectangle(mapInfo, maxSize).take(maxSize)
          completeJsonOk(mapHotelToJson(nearHotels, Map.empty, lang))
        } ~ (parameter('max_size.as[Int] ? 10) & metered("near_hotels")) { maxSize =>
          val nearHotels = hotelIndex.near(hotel, maxSize).toSeq
          completeJsonOk(mapHotelToJson(nearHotels, Map.empty, lang))
        }
      } ~ (path("near_cities") & parameter('max_size.as[Int] ? 5) & metered("near_cities")) { maxSize =>
        val array = new JSONArray()
        directionsStats.getPopularCities(hotel.geoId).take(maxSize).foreach { city =>
          array.put(regionToJson(city, lang))
        }
        completeJsonOk(array)
      } ~ (path("near_countries") & parameter('max_size.as[Int] ? 5) & metered("near_countries")) { maxSize =>
        val array = new JSONArray()
        val near = directionsStats.getNearCountries(hotel.geoId)
        val similar = directionsStats.getSimilarCountries(hotel.geoId)
        (near ++ similar).distinct.take(maxSize).foreach { country =>
          array.put(regionToJson(country, lang))
        }
        completeJsonOk(array)
      } ~ pathPrefix("rooms") {
        roomsOfferSearchRoute(hotel)
      } ~ pathPrefix("tours") {
        toursOfferSearchRoute(hotel)
      } ~ toursOfferSearchRoute(hotel)
    }
  }
}
