package ru.yandex.tours.api.v1.hotel

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.{CacheMetrics, JsonSerialization, ReqAnsLogger}
import ru.yandex.tours.backend.HotelSnippetPreparer
import ru.yandex.tours.backend.HotelSnippetPreparer.{OfferWithBilling, OffersWithInfo, SnippetWithBilling}
import ru.yandex.tours.filter.Filters.snippetFilters
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsVideo
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.hotels.{Hotel, RichHotelSnippet}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.http.UrlLabelerBuilder
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.LabelBuilder
import spray.routing.Directives._
import spray.routing.Route

import scala.concurrent.ExecutionContext

/**
 * Offer related routes:
 * {{{
 * /search?<request>&<lang>&<paging>&<sortBy>&<filters>&request_index
 * /similar?<request>&max_size=3&<lang>
 * }}}
 */
class OfferSearchRouter[S <: Source](routeesContext: RouteesContext,
                                     offerPreparer: HotelSnippetPreparer[S],
                                     tree: Tree,
                                     searchSources: SearchSources[S],
                                     context: SearchType,
                                     geoMappingHolder: GeoMappingHolder,
                                     searchSettingsHolder: SearchSettingsHolder,
                                     hotelsVideo: HotelsVideo,
                                     reqAnsLogger: ReqAnsLogger,
                                     redirectUrl: String,
                                     labelBuilder: LabelBuilder)
                                    (implicit ec: ExecutionContext) extends JsonSerialization with Bindings {

  private val cacheMetrics = new CacheMetrics(context.toString.toLowerCase + "_search")
  override protected def metered(name: String) = super.metered("hotel." + context.toString.toLowerCase + "_" + name)

  def route(hotel: Hotel): Route = {
    (path("search") & metered("search")) {
      searchOffers(hotel)
    } ~ (path("similar") & metered("similar")) {
      searchSimilar(hotel)
    }
  }

  private def searchOffers(hotel: Hotel): Route = {
    (routeesContext.offersSearchRequest(hotel) & lang & paging & sortBy & snippetFilters & requestIndex
      & utm & userIdentifiers) {
      (offerRequest, lang, paging, sortBy, filters, reqIndex, utm, user) =>
        onSuccess(offerPreparer.getOffersInHotel(offerRequest, sortBy, paging, filters, reqIndex == 0)) { result: OffersWithInfo =>
          val statistic = result.statistic

          cacheMetrics.update(reqIndex, result.progress, statistic.total)
          reqAnsLogger.logOffers(reqIndex, offerRequest, paging, filters, sortBy, result, user, context)

          //TODO. This serialization has too much common with serialization is SearchHandler. Join them
          val ans = new JSONObject()
          val labelerBuilder = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder)

          val offers = new JSONArray()
          result.tours.foreach { offer: OfferWithBilling =>
            offers.put(toJson(offer.offer, lang, searchSources, offer.billingObject, labelerBuilder.buildFrom(offer.offer)))
          }
          ans.put("tours", toJson(offers, paging, statistic.found))
          ans.put("progress", toJson(result.progress, statistic.found, statistic.total))
          ans.put("statistic", toJson(statistic))

          val resultInfoJs = toJson(
            result.resultInfo,
            countrySearch = false,
            filtersEnabled = filters.exists(_.nonEmpty),
            allFiltered = statistic.allFiltered
          )
          ans.put("result_info", resultInfoJs)

          val allowedState = new JSONObject()
          result.allowedStates.foreach(_.putToJson(allowedState))
          ans.put("allowed_filters", allowedState)

          val jsonFilters = filtersToJson(filters)
          jsonFilters.put("sort", sortBy.toString.toLowerCase)
          ans.put("filters", jsonFilters)
          ans.put("search_request", toJson(offerRequest, tree, lang))

          completeJsonOk(ans)
        }
    }
  }

  def searchSimilar(hotel: Hotel): Route = {
    (routeesContext.offersSearchRequest(hotel) & parameter('max_size.as[Int] ? 3) & lang & utm & userIdentifiers) {
      (request, count, lang, utm, user) =>
      onSuccess(offerPreparer.getSimilarHotels(request.hotelRequest, hotel, count)) { result =>
        val ans = new JSONArray()
        for ((hotel, snippet, billing, hotelBilling) <- result) {
          val res = new JSONObject()
          res.put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))
          snippet.foreach { s =>
            val snippetWithBilling = SnippetWithBilling(RichHotelSnippet(s, hotel), billing, hotelBilling)
            val labeler = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder).buildFrom(snippetWithBilling.sample)
            val tourJson = snippetToJson(snippetWithBilling, request.hotelRequest.ages, tree, searchSources,
              hotelsVideo, geoMappingHolder, searchSettingsHolder, lang, labeler)
            res.put("tour", tourJson)
          }
          res.put("search_request", toJson(request.copy(hotelId = hotel.id), tree, lang))
          ans.put(res)
        }
        completeJsonOk(ans)
      }
    }
  }
}

