package ru.yandex.tours.api.v1.recommend

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.direction.layout.Layout
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.Prices.{Recommendation, RecommendedDirection}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.WhereToGoRequest
import ru.yandex.tours.model.util.{Paging, SortType}
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.services.RecommendService
import ru.yandex.tours.util.spray._

import scala.collection.JavaConversions._
import scala.concurrent.Future

/**
 * Recommendation related routes:
 * {{{
 * /?<user>&<geo_id>&<lang>
 * /directions?<request>&<user>&<geo_id>&<paging>&<sort_by>&<lang>
 * /directions?<user>&<geo_id>&<paging>&<sort_by>&<lang>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class RecommendHandler(tree: Tree,
                       recommendService: RecommendService,
                       searchType: SearchType) extends HttpHandler with JsonSerialization with Bindings {

  override protected def metered(name: String) =
    super.metered("recommend." + searchType.toString.toLowerCase + "_" + name)

  private val skipHead = parameter('skip_head.as[Boolean] ? true)
  private val commonParams = userIdentifiers & fromOrGeoId & lang & branding

  override lazy val route = {
    (pathEndOrSingleSlash & commonParams & metered("recommend")) {
      (userIdentifiers, geoId, lang, branding) =>
        onSuccess(recommend(userIdentifiers, geoId, SortType.RELEVANCE, branding)) { directions =>
          val arr = new JSONArray()
          directions.take(1).foreach(dir => arr.put(toJson(dir, geoId, lang, tree)))
          val ans = new JSONObject()
            .put("directions", arr)
          completeJsonOk(ans)
        }
    } ~ (path("resorts") & userIdentifiers & fromOrGeoId & lang & parameter("country_id".as[Int])) {
      (userIdentifiers, geoId, lang, countryId) =>
        onSuccess(recommendService.recommendResorts(countryId, userIdentifiers, geoId)) { directions =>
          completeResorts(directions.getDirectionsList, geoId, lang)
        }
    } ~ (path("directions") & whereToGoRequest & commonParams & paging & sortBy & metered("recommend_thematic")) {
      (request, userIdentifiers, geoId, lang, branding, paging, sortBy) =>
        onSuccess(recommendService.recommendDirections(request, userIdentifiers, geoId, sortBy)) { recommendation =>
          val directions = recommendation.getDirectionsList.toSeq
          completeDirections(Some(request), directions, geoId, paging, sortBy, lang)
        }
    } ~ (path("directions") & commonParams & paging & sortBy & skipHead & metered("recommend_directions")) {
      (userIdentifiers, geoId, lang, branding, paging, sortBy, skipHead) =>
        onSuccess(recommend(userIdentifiers, geoId, sortBy, branding)) { directions =>
          val directions2 = if (skipHead) directions.drop(1) else directions
          completeDirections(None, directions2, geoId, paging, sortBy, lang)
        }
    }
  }

  private def recommend(userIdentifiers: UserIdentifiers,
                        userRegion: Int, sortBy: SortType,
                        branding: Option[String]): Future[Seq[RecommendedDirection]] = {
    recommendService.recommendAllDirections(userIdentifiers, userRegion, sortBy, branding)
      .map { _.getDirectionsList.toSeq }
  }

  private def completeResorts(directions: Seq[RecommendedDirection], geoId: Int, lang: Lang) = {
    val layout = Layout.bySize(directions.size)
    val array = toJson(layout, directions, geoId, tree, lang)
    completeJsonOk(array)
  }

  private def completeDirections(request: Option[WhereToGoRequest],
                                 directions: Seq[RecommendedDirection],
                                 geoId: Int,
                                 paging: Paging,
                                 sortBy: SortType,
                                 lang: Lang) = {
    val paged = paging(directions)
    val layout = if (request.isDefined) Layout.staticBySize(paged.size) else Layout.bySize(paged.size)
    val array = toJson(layout, paged, geoId, tree, lang)
    val ans = toJson(array, paging, directions.size)
    val filtersState = request.fold(new JSONObject())(toJson)
    filtersState.put("sort", sortBy.toString.toLowerCase)
    ans.put("filters_state", filtersState)
    completeJsonOk(ans)
  }
}
