package ru.yandex.tours.api.v1.recommend

import akka.actor.ActorRef
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.avia.AviaAirportRecommendations
import ru.yandex.tours.direction.{Directions, DirectionsStats}
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsIndex, HotelsVideo}
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.spray._
import spray.routing.Route

/**
 * {{{
 * /another_cities?<to>&max_size=8&<lang>
 * /another_countries?<to>&max_size=8&<lang>
 * /another_direction?<geo_id>&max_size=6&<lang>
 * /hotels?<to>&max_size=8&<lang>
 * /airport?<to|geo_id>&<from>&<lang>
 * /search_request?<request>&<lang>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */

class RootRecommendHandler(routeesContext: RouteesContext,
                           tree: Tree,
                           hotelsIndex: HotelsIndex,
                           hotelsVideo: HotelsVideo,
                           geoMapping: GeoMappingHolder,
                           searchSettings: SearchSettingsHolder,
                           directions: Directions,
                           directionsStats: DirectionsStats,
                           aviaAirportRecommendations: AviaAirportRecommendations,
                           roomsRecommendHandler: ActorRef,
                           toursRecommendHandler: ActorRef) extends HttpHandler with JsonSerialization with Bindings {

  override protected def metered(name: String) = super.metered("recommend." + name)
  private def maxSize(default: Int) = parameter('max_size.as[Int] ? default)
  private val optTo = parameters('to.as[Int].?)

  override lazy val route: Route =
    (path("another_cities") & optTo & maxSize(8) & lang & metered("another_cities")) {
      (toRegion, maxSize, lang) =>
        val json = new JSONObject()
        val array = new JSONArray()

        for {
          region <- toRegion
          country <- tree.country(region)
        } json.put("country", regionToJson(country, lang))

        val cities = toRegion match {
          case Some(region) => directionsStats.getPopularCities(region)
          case None => directionsStats.getPopularCities()
        }
        cities.take(maxSize).foreach { city =>
          array.put(regionToJson(city, lang))
        }

        json.put("cities", array)
        completeJsonOk(json)
  } ~ (path("another_countries") & optTo & maxSize(8) & lang & metered("another_countries")) {
    (toRegion, maxSize, lang) =>
      val array = new JSONArray()
      val countries = toRegion match {
        case Some(region) =>
          val near = directionsStats.getNearCountries(region)
          val similar = directionsStats.getSimilarCountries(region)
          near ++ similar
        case None =>
          directionsStats.getPopularCountries()
      }
      countries.distinct.take(maxSize).foreach { country =>
        array.put(regionToJson(country, lang))
      }
      completeJsonOk(array)
  } ~ (path("another_direction") & toOrGeoId & maxSize(6) & lang & metered("another_directions")) {
    (geoId, maxSize, lang) =>
      val array = new JSONArray()

      val similar = directionsStats.getSimilarCountries(geoId)
      val popular = directionsStats.getPopularCountries(geoId)
      (similar ++ popular)
        .flatMap(r => directions.get(r.id))
        .distinct
        .take(maxSize)
        .foreach { direction => array.put(directionToJson(direction, lang)) }
      completeJsonOk(array)
  } ~ (path("hotels") & optTo & maxSize(8) & lang) { (to, maxSize, lang) =>
    val array = new JSONArray()
    val hotels = to match {
      case Some(region) =>
        var top = hotelsIndex.topInRegion(region, maxSize).toSeq
        if (top.size < maxSize) {
          for (region <- tree.region(region)) {
            top ++= hotelsIndex.inRectangle(region.boundingBox, maxSize).sortBy(h => geo.distanceInKm(region, h))
          }
        }
        top.distinct.take(maxSize)
      case None => hotelsIndex.getHotelsSorted(Paging(pageSize = maxSize))
    }

    hotels.foreach {
      hotel => array.put(toJson(hotel, hotelsVideo, tree, geoMapping, searchSettings, lang))
    }
    completeJsonOk(array)
  } ~ (path("airport") & toOrGeoId & parameter('from.as[Int]) & lang) { (to, from, lang) =>
    val res = new JSONArray()
    aviaAirportRecommendations.recommend(from, to).foreach {
      airport => res.put(toJson(airport, tree, lang))
    }
    completeJsonOk(res)
  } ~ (path("search_request") & routeesContext.searchRequest(defaultTo = 0) & lang & metered("search_request")) {
    (searchRequest, lang) =>
      val ans = toJson(searchRequest, tree, lang, sortBy = None)
      if (searchRequest.to == 0) {
        ans.remove("to")
      }
      completeJsonOk(ans)
  } ~ pathPrefix("rooms") {
    ctx => roomsRecommendHandler ! ctx
  } ~ pathPrefix("tours") {
    ctx => toursRecommendHandler ! ctx
  } ~ {
    ctx => toursRecommendHandler ! ctx
  }
}
