package ru.yandex.tours.api.v1.reference

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsExport.Fields
import ru.yandex.tours.hotels.HotelsExport.Fields._
import ru.yandex.tours.hotels.{HotelsIndex, HotelsExport, HotelsExportMarshaller, HotelsVideo}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.image.ImageFormats
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.ota.OnlineTourAgencies
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.Tabbed
import ru.yandex.tours.util.spray.{HttpHandler, _}
import spray.http.{MediaTypes, StatusCodes}
import spray.routing.{RequestContext, Route}

import scala.collection.mutable

class OtaHandler(geoMappingHolder: GeoMappingHolder,
                 tree: Tree,
                 hotelsIndex: HotelsIndex,
                 hotelsVideo: HotelsVideo,
                 hotelsExport: HotelsExport,
                 hotelsExportMarshaller: HotelsExportMarshaller,
                 operators: TourOperators,
                 otas: OnlineTourAgencies) extends HttpHandler with Bindings {

  override protected def metered(name: String) = super.metered("reference.ota_" + name)

  private def validTokens: List[String] = {
    otas.agencies.map(_.authToken) ++ Seq(
      "3OGCmdPkzM86" // lookInHotels
    )
  }

  override def route: Route = authorize(validTokens) {
    (path("regions") & metered("regions")) {
      dynamic {
        val geoIds = interestingGeoIds
        val root = tree.root.ensuring(_.isDefined, "Region tree should have root").get
        val dfs = serializingDfs(root, geoIds, mutable.HashSet.empty[Region])
        completePlainJson(dfs.get.toString)
      }
    } ~ (path("operators") & metered("operators")) {
      dynamic {
        val response = new JSONArray()
        operators.getAll.foreach { operator =>
          response.put(new JSONObject()
            .put("name", operator.name)
            .put("id", operator.id)
          )
        }
        completePlainJson(response.toString)
      }
    } ~ (path("hotels") & metered("hotels")) {
      dynamic {
        val hotelFields = Seq(Fields.ID, Fields.RU_NAME, Fields.EN_NAME, Fields.STARS, Fields.REGION_ID, Fields.BOOKING_ID)
        hotelsExportMarshaller.marshallHotelsExport(hotelFields, hotelFilter)
      }
    } ~ (path("video") & metered("video")) {
      dynamic {
        val result = IO.printString { pw =>
          for ((id, video) <- hotelsVideo.all) {
            val hotel = hotelsIndex.getHotelById(id)
            val region = hotel.map(_.geoId).flatMap(tree.region)
            val country = region.flatMap(tree.country)
            pw.println(Tabbed(
              id,
              hotel.fold("")(_.name.ruName),
              region.fold("")(_.name.ruName),
              country.fold("")(_.name.ruName),
              video.preview.inFormat(ImageFormats.orig),
              video.preview.source.getOrElse(""),
              video.videoUrl
            ))
          }
        }
        complete(result)
      }
    }
  }

  private def hotelFilter(field: Field, value: String) = !(field == Fields.TOURS_SEARCH && value == "false")

  private def knownDestinations(ids: Set[Int]) = {
    ids.filter(geoMappingHolder.isKnownDestination(_, SearchType.TOURS))
  }

  private def serializingDfs(region: Region,
                             interestingIds: Set[Int],
                             visited: mutable.HashSet[Region]): Option[JSONObject] = {
    if (visited.contains(region)) return None
    visited += region
    val children = tree.children(region).flatMap(serializingDfs(_, interestingIds, visited))
    if (children.nonEmpty || interestingIds.contains(region.id)) {
      val name = new JSONObject()
      for {
        lang <- Seq(Languages.ru, Languages.en)
        value <- region.name.get(lang)
      } name.put(lang.toString, value)

      val result = new JSONObject()
        .put("name", name)
        .put("id", region.id)
      if (geoMappingHolder.isDepartureCity(region.id)) result.put("is_departure", true)
      if (geoMappingHolder.isKnownDestination(region.id)) result.put("is_destination", true)
      if (region.isCountry) result.put("is_country", true)
      if (children.nonEmpty) {
        val childrenJson = new JSONArray()
        children.foreach(childrenJson.put)
        result.put("children", childrenJson)
      }
      Some(result)
    } else {
      None
    }
  }

  private def interestingGeoIds = {
    val exportIs = hotelsExport.read(Seq(Fields.REGION_ID), hotelFilter)
    try {
      val hotelGeoIds = scala.io.Source.fromInputStream(exportIs).getLines().map(_.toInt)
      hotelGeoIds.toSet ++
        knownDestinations(geoMappingHolder.countryGeoIds) ++
        knownDestinations(geoMappingHolder.cityGeoIds) ++
        geoMappingHolder.departuresGeoIds
    } finally {
      exportIs.close()
    }
  }

  private def completePlainJson(data: String): RequestContext => Unit = {
    respondWithMediaType(MediaTypes.`application/json`) {
      complete(StatusCodes.OK, data)
    }
  }
}
