package ru.yandex.tours.api.v1.reference

import akka.actor.ActorRef
import org.joda.time.LocalDate
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.agencies.AgenciesIndex
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.direction.{CountryResortsService, DirectionWithHotelCount, RegionWithHotelCount}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsExport.Fields
import ru.yandex.tours.hotels.{HotelsExportMarshaller, HotelsIndex}
import ru.yandex.tours.metro.MetroHolder
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.filter.hotel.ChannelManagerFilter
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.operators.{HotelProviders, SearchSources, TourOperators}
import ru.yandex.tours.ota.OnlineTourAgencies
import ru.yandex.tours.resorts.Brandings
import ru.yandex.tours.services.CalendarService
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.spray.{HttpHandler, _}
import spray.http.StatusCodes
import spray.routing.Route

import scala.collection.JavaConverters._
import scala.util.{Failure, Success, Try}

/**
 * Static resources routes:
 * {{{
 * /departures?<lang>
 * /operators
 * /hotel_providers
 * /destinations?<lang>
 * /metros?<lang>&geo_id
 * /flight_matrix
 * /agency_cities?<lang>
 * /partners
 * /country_info>?<lang>&<to>&<size>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class ReferenceHandler(tree: Tree,
                       mapping: GeoMappingHolder,
                       tourOperators: TourOperators,
                       hotelProviders: HotelProviders,
                       otas: OnlineTourAgencies,
                       hotelsIndex: HotelsIndex,
                       metroHolder: MetroHolder,
                       calendarService: CalendarService,
                       countryResortsService: CountryResortsService,
                       agenciesIndex: AgenciesIndex,
                       brandings: Brandings,
                       hotelsExportMarshaller: HotelsExportMarshaller,
                       otaHandler: ActorRef) extends HttpHandler with Bindings with JsonSerialization {

  override protected def metered(name: String) = super.metered("reference." + name)

  private def getRegions(regionIds: Iterable[Int], lang: Lang) = {
    val regions = regionIds.flatMap(tree.region)
    val res = new JSONArray()
    regions.foreach(region => {
      res.put(new JSONArray()
        .put(region.id)
        .put(region.name(lang))
        .put(region.genitive)
      )
    })
    res
  }

  private def serializeSearchSources[S <: Source](sources: SearchSources[S]) = {
    val operators = new JSONArray()
    sources.getAll
      .filterNot(_.contains(Partners.kurortExpress))
      .foreach(operator => {
        operators.put(new JSONObject()
          .put("name", operator.name)
          .put("id", operator.id.toString)
          .put("is_cmhotel", if (operator.code == "view_on_site") true else null)
        )
      })
    operators
  }

  override def route: Route =
    pathPrefix("ota") {
      http => otaHandler ! http
    } ~ (path("departures") & lang & metered("departures")) { lang =>
      completeJsonOk(getRegions(mapping.departuresGeoIds, lang))
    } ~ (path("operators") & metered("operators")) {
      dynamic {
        completeJsonOk(serializeSearchSources(tourOperators))
      }
    } ~ (path("hotel_providers") & metered("hotel_providers")) {
      dynamic {
        completeJsonOk(serializeSearchSources(hotelProviders))
      }
    } ~ (path("region") & parameter('id.as[Int]) & lang) { (id, lang) =>
      tree.region(id) match {
        case Some(region) => completeJsonOk(regionToJson(region, lang))
        case None => completeJsonError(StatusCodes.NotFound, s"Region $id not found")
      }
    } ~ (path("destinations") & lang & metered("destinations")) { lang =>
      completeJsonOk(getRegions(mapping.cityGeoIds | mapping.countryGeoIds, lang))
    } ~ (path("metros") & lang & parameter('geo_id.as[Int]) & metered("metros")) { (lang, geoId) =>
      val metros = metroHolder.byGeoId(geoId)
      val ar = new JSONArray()
      metros.foreach(m => ar.put(toJson(m, lang)))
      completeJsonOk(ar)
    } ~ (path("flight_matrix") & metered("flight_matrix")) {
      dynamic {
        onSuccess(calendarService.getFlightMatrix) { matrix =>
          val ans = new JSONObject()
          for (row <- matrix.getRowList.asScala) {
            val ar = new JSONArray()
            for (to <- row.getToList.asScala) {
              ar.put(to)
            }
            ans.put(row.getFrom.toString, ar)
          }
          completeJsonOk(ans)
        }
      }
    } ~ (path("agency_cities") & lang & metered("agency_cities")) { lang =>
      val regions = agenciesIndex.agencies.map(_.agency.geoId).toSeq.distinct
      val array = new JSONArray()
      for {
        id <- regions
        region <- tree.region(id)
      } array.put(new JSONObject()
        .put("id", region.id)
        .put("name", region.name(lang))
      )
      completeJsonOk(getRegions(regions, lang))
    } ~ (path("partners") & metered("partners")) {
      dynamic {
        val ans = new JSONArray()
        def serialize(id: Int, code: String, name: String, `type`: String) = new JSONObject()
          .put("code", code)
          .put("name", name)
          .put("type", `type`)
          .put("id", id)

        tourOperators.getAll.foreach {
          to => ans.put(serialize(to.id, to.code, to.name, "tour_operator"))
        }
        otas.agencies.foreach {
          ota => ans.put(serialize(ota.id, ota.billingId, ota.name, "ota"))
        }
        hotelProviders.getAll.foreach {
          hp => ans.put(serialize(hp.id, hp.code, hp.name, "hotel_provider"))
        }
        completeJsonOk(ans)
      }
    } ~ (path("active_branding") & optDate("date") & metered("active_branding")) { date =>
      val realDate = date.getOrElse(LocalDate.now)
      val arr = new JSONArray()
      brandings.getActive(realDate).foreach(arr.put)
      completeJsonOk(arr)
    } ~ (path("country_info") & lang & parameters('to.as[Int], 'size.as[Int] ? 6) & metered("country_info")) { (lang, geoId, size) =>
      val all = countryResortsService.getResorts(geoId)
      val top = countryResortsService.getTopResorts(geoId, size)
      val ans = new JSONObject()
      val topArray = new JSONArray()
      top.foreach { case DirectionWithHotelCount(direction, count) =>
        topArray.put(new JSONObject()
          .put("region", regionToJson(direction.region, lang))
          .put("image", toJson(direction.mainImage))
          .put("count", count)
        )
      }
      val allArray = new JSONArray()
      all.toSeq.sortBy(_.region.name(lang)).foreach { case RegionWithHotelCount(region, count) =>
        allArray.put(new JSONObject()
          .put("region", regionToJson(region, lang))
          .put("count", count)
        )
      }
      ans.put("top", topArray).put("all", allArray)
      completeJsonOk(ans)
    } ~ (path("hotels") & array("fields", isEmptyOk = true) & metered("hotels")) { rawFields =>
      Try(rawFields.map(Fields.withName)) match {
        case Success(fields) =>
        hotelsExportMarshaller.marshallHotelsExport(if (fields.isEmpty) Fields.values else fields, { case _ => true })
        case Failure(e) => complete(StatusCodes.BadRequest, s"Can not parse hotels export fields: ${e.getMessage}")
      }
    } ~ (path("geobase") & metered("geobase")) {
      dynamic {
        val result = IO.printBytes { pw =>
          tree.regions.foreach(r => pw.println(r.toTsv))
        }
        complete(result)
      }
    } ~ (path("filter_channel_managers") & intArray("ids", isEmptyOk = true) & metered("filter_channel_managers")) { ids =>
      val array = new JSONArray()
      val cmIds = hotelsIndex.filter(ids, Seq(ChannelManagerFilter), None)
      hotelsIndex.getHotelsById(cmIds).mapValues { hotel =>
        hotel.partnerIds.find(pInfo => Partners.isChannelManager(pInfo.partner)).map(_.travelId)
      }.foreach {
        case (id, Some(cmId)) => array.put(new JSONObject().put("id", id).put("cmId", cmId))
        case (id, None) => log.warning(s"Failed to resolve channel manager id for hotel $id")
      }
      completeJsonOk(array)
    }
}
