package ru.yandex.tours.api.v1.resort

import org.joda.time.DateTimeZone
import org.json.{JSONArray, JSONObject}
import play.api.libs.json.Json
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.currency.CurrencyRates
import ru.yandex.tours.direction.Directions
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.base.{Region => GeobaseRegion}
import ru.yandex.tours.model.BaseModel
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.resorts.Implicits._
import ru.yandex.tours.resorts.ReviewsInformation.ResortStats
import ru.yandex.tours.resorts._
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.spray._
import spray.http.StatusCodes

import scala.concurrent.duration._

/**
 * Created by Evgeny Zhoga on 02.12.15.
 *
 * TODO: make documentations
 */
class ResortSkiHandler(skiResorts: SkiResorts,
                       directions: Directions,
                       currencyRates: CurrencyRates,
                       tree: Tree,
                       extDataService: ExtDataService
                      ) extends HttpHandler with Bindings with JsonSerialization {

  private def toRegion(r: GeobaseRegion, lang: Lang): Region = {
    Region(r.id, Some(r.name(lang)), Some(r.genitive), Some(r.accusative), Some(r.preposition), Some(r.locative))
  }

  private def prepare(s: Ski, lang: Lang) = {
    val span = tree.region(s.id)
      .map(r => Coordinates(
        latSpan = r.boundingBox.latSpan,
        lonSpan = r.boundingBox.lonSpan,
        latitude = r.boundingBox.latCenter,
        longitude = r.boundingBox.lonCenter
      ))

    s.copy(
      location = Some(Location(
        country = tree.country(s.id).map(r => toRegion(r, lang)),
        coordinates = span,
        region = tree.region(s.id).map(r => toRegion(r, lang))
      )),
      subregions =
        if (s.isMetaRegion && s.subregionsIds.isDefined) {
          Some(s.subregionsIds.get.flatMap(tree.region).map(r => toRegion(r, lang)))
        } else None,
      regions = s.regions.map(r => tree.region(r.id).map(r => toRegion(r, lang)).getOrElse(r))
    )
  }

  override protected def metered(name: String) = super.metered("resorts." + name)

  override val route = pathPrefix("ski")  {
    (pathEndOrSingleSlash & paging & lang & metered("list")) {
      (paging, lang) =>
        val res = new JSONObject()
        val paged = paging(skiResorts.resorts.map(_.ski)).map(prepare(_, lang))
        val json = toJson(new JSONArray(camelCaseToUnderscoreJs(Json.toJson(paged)).toString()), paging, skiResorts.resorts.size)
        res.put("resorts", json)
        completeJsonOk(res)
    } ~  (path("region" / IntNumber ~ Slash.?) & paging & lang & metered("regions")) {
      (regionId, paging, lang) =>
        val resorts = skiResorts.getByRegion(regionId)
        val res = new JSONObject()
        val paged = paging(resorts.map(_.ski)).map(prepare(_, lang))
        val json = toJson(new JSONArray(camelCaseToUnderscoreJs(Json.toJson(paged)).toString()), paging, resorts.size)
        res.put("resorts", json)
        completeJsonOk(res)
    } ~ (path(IntNumber / "reviews" ~ Slash.?)  & paging & metered("reviews")) {
      (resortId, paging) =>
        skiResorts.resorts.find(_.ski.id == resortId) match {
          case None =>
            completeJsonError(StatusCodes.NotFound, s"Ski Resort with [id:$resortId] is not found")
          case Some(resort) if resort.reviewsInformation.isDefined =>
            val reviewsInformation = resort.reviewsInformation.get
            val paged = paging(reviewsInformation.reviews)
            val reviewCount = reviewsInformation.reviews.size
            val res = toJson(new JSONArray(camelCaseToDashJs(Json.toJson(paged)).toString()), paging, reviewCount)
            completeJsonOk(res)
          case _ =>
            val res = toJson(new JSONArray(), paging, 0)
            completeJsonOk(res)
        }
    } ~ (path(IntNumber / "statistics" ~ Slash.?) & metered("statistics")) {
      (resortId) =>
        skiResorts.resorts.find(_.ski.id == resortId) match {
          case None =>
            completeJsonError(StatusCodes.NotFound, s"Ski Resort with [id:$resortId] is not found")
          case Some(resort) if resort.reviewsInformation.isDefined =>
            val resortStats = resort.reviewsInformation.get.resortStats
            completeJsonOk(new JSONObject(camelCaseToDashJs(Json.toJson(resortStats)).toString()))
          case _ =>
            val resortStats = ResortStats(resortId, 0, 0, None, List.empty)
            completeJsonOk(new JSONObject(camelCaseToDashJs(Json.toJson(resortStats)).toString()))
        }
    } ~ (path(IntNumber ~ Slash.?) & lang & metered("get")) {
      (resortId, lang) =>
        skiResorts.get(resortId).map(_.ski).map(prepare(_, lang)) match {
          case Some(result) =>
            completeJsonOk(new JSONObject(camelCaseToUnderscoreJs(Json.toJson(result)).toString()))
          case None =>
            completeJsonError(StatusCodes.NotFound, s"Ski Resort with [id:$resortId] is not found")
        }
    }
  } ~ (path("jordan") & lang) { lang =>
    val data = IO.readLines(extDataService.readData(DataTypes.jordan)).mkString("\n")
    val json = new JSONObject(data)
    for (jordan <- directions.get(10535)) {
      json.put("name", jordan.region.name(lang))
      json.put("main_image", toJson(jordan.mainImage))
      for (tz <- jordan.region.timeZone) {
        val now = System.currentTimeMillis()
        val moscow = DateTimeZone.forID("Europe/Moscow")
        val offset = jordan.region.timeZone.get.getOffset(now) - moscow.getOffset(now)
        val hoursDiff = (offset.millis.toMinutes.toDouble / 30d).round.toDouble / 2
        json.put("timezone_diff", hoursDiff)
      }
      val currencyRate = currencyRates.getRate(BaseModel.Currency.JOD, BaseModel.Currency.RUB)
      json.getJSONObject("currency").put("rate", currencyRate.doubleValue())
      json.put("coordinates", toJson(jordan.region.boundingBox))
      json.put("country", regionToJson(jordan.region, lang))
      json.put("capital", regionToJson(11494, tree, lang))
      json.put("main_city", regionToJson(20792, tree, lang))
    }
    completeJsonOk(json)
  }
}
