package ru.yandex.tours.api.v1.search

import akka.actor.Props
import ru.yandex.tours.api.v1.view.{FlightAlternativesView, FlightSearchResultView}
import ru.yandex.tours.api.{JsonSerialization, ReqAnsLogger}
import ru.yandex.tours.avia.{Airports, AviaAirportRecommendations}
import ru.yandex.tours.geo.Departures
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.services.FlightSearchService
import ru.yandex.tours.util.spray._
import spray.routing.Route

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 15.02.16
 */
class FlightSearchHandler(routeesContext: RouteesContext,
                          flightSearchService: FlightSearchService,
                          aviaAirportRecommendations: AviaAirportRecommendations,
                          departures: Departures,
                          airports: Airports,
                          tree: region.Tree,
                          reqAnsLogger: ReqAnsLogger) extends HttpHandler with Bindings with JsonSerialization {

  override protected def metered(name: String) = super.metered("search."  + name)

  override def route: Route = {
    (pathEndOrSingleSlash & routeesContext.searchRequest & parameters('from_override.as[Int].?, 'airport_id.?) &
      lang & requestIndex & userIdentifiers & metered("flights")) {
      (searchRequest, fromOverride, airportOverride, lang, requestIndex, user) ⇒
        val departures = this.departures.getDepartures(searchRequest.from, searchRequest.to)
        val destinations = aviaAirportRecommendations.recommendMany(searchRequest.from, searchRequest.to)
        val airport = airportOverride.orElse(destinations.headOption.map(_.city.id))
          .getOrElse("g" + searchRequest.to)

        val from = fromOverride.getOrElse(searchRequest.from)

        val request = FlightSearchRequest(searchRequest, airport).withFrom(from)
        val canStartRequest = requestIndex == 0

        val flights = flightSearchService.searchFlights(request, canStartRequest)
        val flightMatrix = flightSearchService.getFlightMatrix(request, departures, destinations.map(_.city.id))
        onSuccess(flights zip flightMatrix) { case (result, matrix) =>
          val flightMatrix = new FlightAlternativesView(request, departures, destinations, matrix)
          val view = new FlightSearchResultView(request, flightMatrix, result)

          reqAnsLogger.logFlights(requestIndex, request, result, user)
          completeJsonOk(view.asJson(airports, tree, lang))
        }
    }
  }
}

object FlightSearchHandler {
  def props(routeesContext: RouteesContext,
            flightSearchService: FlightSearchService,
            aviaAirportRecommendations: AviaAirportRecommendations,
            departures: Departures,
            airports: Airports,
            tree: region.Tree,
            reqAnsLogger: ReqAnsLogger): Props = {
    Props(new FlightSearchHandler(routeesContext, flightSearchService,
      aviaAirportRecommendations, departures, airports, tree, reqAnsLogger))
  }
}
