package ru.yandex.tours.api.v1.search

import org.joda.time.LocalDate
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.v1.view.SearchResultView
import ru.yandex.tours.api.{CacheMetrics, JsonSerialization, ReqAnsLogger}
import ru.yandex.tours.backend.HotelSnippetPreparer.SnippetWithBilling
import ru.yandex.tours.backend.{HotelMinPricePreparer, HotelSnippetPreparer}
import ru.yandex.tours.filter.Filters
import ru.yandex.tours.geo.Departures
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsIndex, HotelsVideo}
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.filter.hotel.SearchTypeFilter
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.util.{SortType, proto}
import ru.yandex.tours.model.{Languages, Source}
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.http.UrlLabelerBuilder
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.LabelBuilder


import scala.concurrent.Future

/**
 * Search related handlers:
 * {{{
 *   /map_info?<request>&<map_rectangle>&<lang>
 *   /?<request>&<paging>&<sort_by>&<filters>&<request_index>&<lang>
 *   /another?<request>&<snippet_count>&<lang>
 *   /top?<request>&<size>&<lang>
 *   /hotels?<map_rectangle>&<lang>&<max_size>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class SearchHandler[T <: Source](routeesContext: RouteesContext,
                                 searchService: HotelSnippetPreparer[T],
                                 minPricePreparer: HotelMinPricePreparer,
                                 tree: Tree,
                                 geoMappingHolder: GeoMappingHolder,
                                 searchSettings: SearchSettingsHolder,
                                 hotelsIndex: HotelsIndex,
                                 departures: Departures,
                                 searchSources: SearchSources[T],
                                 cacheMetricName: String,
                                 searchType: SearchType,
                                 hotelsVideo: HotelsVideo,
                                 reqAnsLogger: ReqAnsLogger,
                                 redirectUrl: String,
                                 labelBuilder: LabelBuilder
                                ) extends HttpHandler with Bindings with JsonSerialization {

  private val cacheMetrics = new CacheMetrics(cacheMetricName)
  override protected def metered(name: String) = super.metered("search." + searchType.toString.toLowerCase + "_" + name)

  private val testUsers = Set(
    "vdolbilov",
    "margarita.pyartel",
    "vladimirgorovoy"
  )

  private def isUnderExperiment(user: UserIdentifiers, experiments: Set[String]): Boolean = {
    experiments.exists(_.startsWith("HOTELS-2174")) || user.login.exists(testUsers.contains)
  }

  private def mapInfo(searchRequest: HotelSearchRequest, lang: Lang) =
    (path("map_info") & mapRectangle & Filters.filters & metered("map_info")) { (mapInfo, filters) =>
      onSuccess(searchService.getMapInfo(searchRequest, filters, mapInfo)) {
        hotels => completeJsonOk(mapHotelToJson(hotels, Map.empty, lang))
      }
    }

  private def search(request: HotelSearchRequest, lang: Lang) = {
    (pathEndOrSingleSlash & paging & sortBy & Filters.filters & requestIndex & utm & experiments &
      userIdentifiers & metered("search")) { (paging, sortBy, filters, reqIndex, utm, experiments, user) =>
        val newSort =
          if (isUnderExperiment(user, experiments) && sortBy == SortType.RELEVANCE) SortType.VISITS
          else sortBy
        onSuccess(searchService.getHotelSnippets(request, paging, newSort, filters, reqIndex == 0, user)) { result =>
          val view = SearchResultView(
            request,
            filters,
            sortBy,
            lang,
            paging,
            result
          )

          val statistic = result.statistic
          cacheMetrics.update(reqIndex, result.progress, statistic.total)
          reqAnsLogger.logHotels(reqIndex, user, request, paging, filters, sortBy, result, searchType)

          val labelerBuilder = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder)
          val json = view.asJson(tree, searchSources, hotelsVideo, geoMappingHolder, searchSettings, labelerBuilder)
          completeJsonOk(json)
        }
    }
  }

  private def searchAnother(searchRequest: HotelSearchRequest, lang: Languages.Value) = {
    (path("another") & parameter("snippet_count".as[Int] ? 6) & utm & userIdentifiers & metered("another")) {
      (snippetCount, utm, user) =>
      val labelerBuilder = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder)
      val dateRequests = getNearDatesOrNightsRequests(searchRequest)
      val fromRequests = getAnotherFromRequest(searchRequest)
      val response = findFallback(dateRequests, snippetCount)
        .zip(findFallback(fromRequests, snippetCount))
        .flatMap({ case (dateFallback, fromFallback) =>
          if (dateFallback.isEmpty && fromFallback.isEmpty) {
            getCountryFallback(searchRequest, snippetCount, lang, labelerBuilder)
          } else {
            val ans = new JSONObject()
            serializeFallback(dateFallback, "near_date", ans, snippetCount, lang, labelerBuilder)
            serializeFallback(fromFallback, "near_from", ans, snippetCount, lang, labelerBuilder)
            Future.successful(ans)
          }
        })
      onSuccess(response)(completeJsonOk)
    }
  }

  private def searchTop(searchRequest: HotelSearchRequest, lang: Languages.Value) = {
    (path("top") & parameter("size".as[Int] ? 6) & utm & userIdentifiers & metered("top")) { (size, utm, user) =>
      onSuccess(searchService.getFromLongCache(searchRequest, size)) { snippets =>
        val ans = new JSONObject()
        ans.put("search_request", toJson(searchRequest, tree, lang, None))
        val array = new JSONArray()
        if (snippets.isEmpty) {
          for {
            hotel <- hotelsIndex.topInRegion(searchRequest.to, size, SearchTypeFilter(searchType))
          } {
            val hotelJson = toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettings, lang)
            array.put(new JSONObject().put("hotel", hotelJson))
          }
        } else {
          for {
            snippet <- snippets
          } {
            val labeler = UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder).buildFrom(snippet.sample)
            val json = snippetToJson(snippet, searchRequest.ages, tree,
              searchSources, hotelsVideo, geoMappingHolder, searchSettings, lang, labeler)
            array.put(json)
          }
        }
        ans.put("list", array)
        completeJsonOk(ans)
      }
    }
  }

  private val searchHotels = path("hotels") {
    (mapRectangle & routeesContext.searchRequest(0) & lang & parameter('max_size.as[Int] ? 200)
      & parameter('hotel_id.as[Int].?) & metered("hotels")) {
      (mapInfo, request, lang, maxSize, hotelId) =>
        val hotels = (hotelsIndex.inRectangle(mapInfo, maxSize) ++ hotelId.flatMap(hotelsIndex.getHotelById)).distinct
        onSuccess(minPricePreparer.getHotelsMinPrice(request, hotels)) { minPrices =>
          completeJsonOk(mapHotelToJson(hotels, minPrices, lang))
        }
    }
  }

  override val route =
    (routeesContext.searchRequest & lang) { (searchRequest, lang) =>
      mapInfo(searchRequest, lang) ~
      search(searchRequest, lang) ~
      searchAnother(searchRequest, lang) ~
      searchTop(searchRequest, lang)
    } ~ {
      searchHotels
    }

  private def getNearDatesOrNightsRequests(searchRequest: HotelSearchRequest) = {
    val requests = for {
      dateShift <- -2 to 2
      nightsShift <- -2 to 2
      nights = Math.max(3, searchRequest.nights + 3 * nightsShift)
      date = searchRequest.when.plusDays(3 * dateShift)
      if date.isAfter(LocalDate.now())
    } yield searchRequest.copy(nights = nights, when = date, flexWhen = true, flexNights = true)
    requests.distinct
  }

  private def getAnotherFromRequest(searchRequest: HotelSearchRequest) = {
    departures.getDepartures(searchRequest.from, searchRequest.to)
      .filter(_ != searchRequest.from)
      .map(fallbackFrom => searchRequest.copy(from = fallbackFrom))
  }

  private def getCountryFallback(searchRequest: HotelSearchRequest, snippetCount: Int, lang: Lang,
                                 labelerBuilder: UrlLabelerBuilder) = {
    val country = tree.country(searchRequest.to)
    val request = country.filter(_.id != searchRequest.to).map(region => searchRequest.copy(to = region.id))
    for {
      countryFallback <- findFallback(request.toIterable, snippetCount)
    } yield {
      val ans = new JSONObject()
      serializeFallback(countryFallback, "country", ans, snippetCount, lang, labelerBuilder)
      ans
    }
  }

  private def serializeFallback(fallback: Option[(HotelSearchRequest, Seq[SnippetWithBilling])],
                                name: String,
                                ans: JSONObject,
                                snippetCount: Int,
                                lang: Lang,
                                labelerBuilder: UrlLabelerBuilder) {
    fallback.foreach({ case (request, snippets) =>
      val array = new JSONArray()
      for (snippet <- snippets.take(snippetCount)) {
        val snippetJson = snippetToJson(snippet, request.ages, tree, searchSources,
          hotelsVideo, geoMappingHolder, searchSettings, lang, labelerBuilder.buildFrom(snippet.sample))
        array.put(snippetJson)
      }
      val x = new JSONObject()
        .put("list", array)
        .put("search_request", toJson(request, tree, lang, None))
        .put("start_date_from", proto.toLocalDate(snippets.map(_.snippet.snippet.getDateMin).min))
        .put("start_date_to", proto.toLocalDate(snippets.map(_.snippet.snippet.getDateMax).max))
        .put("nights_from", snippets.map(_.snippet.snippet.getNightsMin).min)
        .put("nights_to", snippets.map(_.snippet.snippet.getNightsMax).max)
      ans.put(name, x)
    })
  }

  private def findFallback(requests: Iterable[HotelSearchRequest], size: Int) = {
    val futures = requests.map(request => searchService.getFromLongCache(request, size).map(t => (request, t)))
    Futures.first(futures)(_._2.nonEmpty)
  }

}
