package ru.yandex.tours.api.v1.search

import akka.actor.Props
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.api.v1.view.TransferSearchResultView
import ru.yandex.tours.avia.{Airports, AviaAirportRecommendations}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.services.TransferSearchService
import ru.yandex.tours.util.spray._
import spray.routing.Route

/**
  * Created by asoboll on 28.04.16.
  */
class TransferSearchHandler(routeesContext: RouteesContext,
                            transferSearchService: TransferSearchService,
                            aviaAirportRecommendations: AviaAirportRecommendations,
                            tree: region.Tree,
                            airports: Airports,
                            hotelsIndex: HotelsIndex) extends HttpHandler with Bindings with JsonSerialization {

  override protected def metered(name: String) = super.metered("search."  + name)

  override def route: Route = {
    (pathEndOrSingleSlash & parameters('airport_id.?, 'hotel_id.as[Int], 'only_cheapest.as[Boolean].?)
      & lang & requestIndex & metered("transfers")) { (airportOverride, hotelId, onlyCheapestOpt, lang, requestIndex) =>

      withHotel(hotelsIndex, hotelId) { hotel =>
        (routeesContext.searchRequestWithDestination(hotel.geoId) & userIp) { (searchRequest, userIp) =>

          lazy val destination = aviaAirportRecommendations.recommend(searchRequest.from, searchRequest.to).map(_.city.id)
          val airport = airportOverride.orElse(destination).getOrElse("g" + searchRequest.to)

          val request = TransferSearchRequest(searchRequest, hotelId, airport, userIp)
          val canStartRequest = requestIndex == 0

          val transfersFuture = transferSearchService.searchTransfers(request, canStartRequest)
          val response = transfersFuture.map(new TransferSearchResultView(request, _))

          val onlyCheapest = onlyCheapestOpt.getOrElse(true)
          onSuccess(response) { view => completeJsonOk(view.asJson(tree, lang, onlyCheapest, airports)) }
        }
      }
    }
  }
}

object TransferSearchHandler {
  def props(routeesContext: RouteesContext,
            flightSearchService: TransferSearchService,
            aviaAirportRecommendations: AviaAirportRecommendations,
            tree: region.Tree,
            airports: Airports,
            hotelsIndex: HotelsIndex): Props = {
    Props(new TransferSearchHandler(routeesContext, flightSearchService, aviaAirportRecommendations,
      tree, airports, hotelsIndex))
  }
}
