package ru.yandex.tours.api.v1.statistic

import akka.actor.ActorRef
import org.joda.time.{Days, LocalDate}
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.calendar.Calendar.FlightDay
import ru.yandex.tours.operators.TourOperatorsState
import ru.yandex.tours.services.CalendarService
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.spray._

/**
 * Statistic related routes:
 * {{{
 * /min_prices?<request>&<lang>&size=29
 * /tour_operators_state
 * /avia?to&from&nights&exact=false&when
 * /calendar?<request>&<lang>
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class RootStatisticHandler(roomsStatisticHandler: ActorRef,
                           toursStatisticHandler: ActorRef,
                           routeesContext: RouteesContext,
                           calendarService: CalendarService,
                           tourOperatorsState: TourOperatorsState)
  extends HttpHandler with Bindings with JsonSerialization {

  override protected def metered(name: String) = super.metered("statistics." + name)

  override val route = (path("tour_operators_state") & metered("tour_operators_state")) {
    onSuccess(tourOperatorsState.getAvailability) { operator2availability =>
      val ans = new JSONArray()
      operator2availability.foreach { case (operator, availability) =>
        ans.put(new JSONObject()
          .put("id", operator.id)
          .put("name", operator.name)
          .put("total", availability.total)
          .put("failures", availability.failures)
          .put("is_alive", availability.isAlive))
      }
      completeJsonOk(ans)
    }
  } ~ (path("calendar") & routeesContext.searchRequest & lang & metered("calendar")) { (searchRequest, lang) =>
    val flightDays = calendarService.getHasFlights(searchRequest.from, searchRequest.to).map { entries =>
      entries.map { e =>
        FlightDay(e.getWhen.toLocalDate, e.getNights)
      }
    }

    onSuccess(flightDays) { rawFlightDays =>
      val flightDays = rawFlightDays.filter(fd => isNearFlightDay(searchRequest.when, fd.when))

      var nights = searchRequest.nights
      var when = searchRequest.when

      val response = new JSONObject()
      val jsonCalendar = new JSONObject()
      val availableNights = new JSONArray()

      flightDays.map(_.nights).distinct.foreach(availableNights.put)
      if (flightDays.nonEmpty) {
        // Just find nearest day, and if "when" is equal, than compare by nights
        val nearestFd = flightDays.minBy(fd => Days.daysBetween(fd.when, when).getDays.abs * 100 + (nights - fd.nights).abs)
        when = nearestFd.when
        nights = nearestFd.nights
      }
      flightDays.groupBy(_.when).foreach { case (date, fds) =>
        val nightsAr = new JSONArray()
        fds.map(_.nights).distinct.foreach(nightsAr.put)
        jsonCalendar.put(date.toString, nightsAr)
      }
      response.put("calendar", new JSONObject()
        .put("flight_days", jsonCalendar)
        .put("available_nights", availableNights)
      )
      response.put("nights", searchRequest.nights)
      completeJsonOk(response)
    }
  } ~ pathPrefix("tours") {
    ctx => toursStatisticHandler ! ctx
  } ~ pathPrefix("rooms") {
    ctx => roomsStatisticHandler ! ctx
  } ~ {
    ctx => toursStatisticHandler ! ctx
  }

  private def isNearFlightDay(when: LocalDate, other: LocalDate) = (months(when) - months(other)).abs < 3

  private def months(when: LocalDate) = when.getMonthOfYear + when.getYear * 12
}
