package ru.yandex.tours.api.v1.statistic

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.calendar.CalendarUtils
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.services.{CalendarService, MinPriceService}
import ru.yandex.tours.util.spray._

import scala.collection.JavaConversions._
import scala.concurrent.Future

/**
 * Statistic related routes:
 * {{{
 * /min_prices?<request>&<lang>&size=29
 * /avia?to&from&nights&exact=false&when
 * /best_price?<request>&<lang> - returns best price and search request near to request
 * }}}
 *
 * @author berkut@yandex-team.ru
 */
class StatisticHandler[S <: Source](routeesContext: RouteesContext,
                                    minPriceService: MinPriceService,
                                    urlBuilder: UrlBuilder,
                                    tree: Tree,
                                    calendarService: CalendarService,
                                    searchSources: SearchSources[S],
                                    searchType: SearchType) extends HttpHandler with Bindings with JsonSerialization {
  private val utmMark = UtmMark.empty.withSource("avia")

  override protected def metered(name: String) =
    super.metered("statistics." + searchType.toString.toLowerCase + "_" + name)

  override val route = (path("min_prices") & metered("min_prices")) {
    (routeesContext.searchRequest & lang & parameter('size.as[Int] ? MinPriceService.DEFAULT_GRAPH_LENGTH)) {
      (searchRequest, lang, graphLength) => {
        onSuccess(minPriceService.getMinPriceGraph(searchRequest, graphLength)) { originalGraph =>
          if (searchType == SearchType.TOURS) {
            onSuccess(CalendarUtils.mergeGraphWithCalendar(originalGraph, calendarService, searchSources)) {
              graph => completeJsonOk(toJson(graph, tree, lang))
            }
          } else {
            completeJsonOk(toJson(originalGraph, tree, lang))
          }
        }
      }
    }
  } ~ (path("avia") & metered("avia")) {
    (intArray("to") & parameters('from.as[Int])) {
      (tos, from) => {
        val futures = tos.map(to => minPriceService.getDirectionMinPrice(from, to))
        onSuccess(Future.sequence(futures)) { bestPrices =>
          val ar = new JSONArray()
          bestPrices.flatten.foreach(bp => ar.put(toJson(bp, urlBuilder, utmMark)))
          completeJsonOk(ar)
        }
      }
    }
  } ~ (path("best_price") & metered("best_price") & routeesContext.searchRequest & lang) { (sr, lang) =>
    onSuccess(minPriceService.getMinPriceGraph(sr, MinPriceService.DEFAULT_GRAPH_LENGTH)) { graph =>
      val minPrices = graph.getPriceEntityList.zipWithIndex.filter(_._1.getMinPrice > 0)
      var resultRequest = sr
      val ans = new JSONObject()
      if (minPrices.nonEmpty) {
        val baseDate = sr.when
        val (bestPrice, index) = minPrices.minBy(_._1.getMinPrice)
        resultRequest = sr.copy(when = baseDate.plusDays(index - graph.getIndex))
        ans.put("best_price", bestPrice.getMinPrice)
      }
      ans.put("search_request", toJson(resultRequest, tree, lang, None))
      completeJsonOk(ans)
    }
  }
}
