package ru.yandex.tours.api.v1.subscription

import ru.yandex.tours.model.filter._
import ru.yandex.tours.model.search.SearchDates
import ru.yandex.tours.model.subscriptions.Subscription
import ru.yandex.tours.model.subscriptions.Subscription.{Hotel, Direction}
import ru.yandex.tours.util.spray._
import ru.yandex.vertis.subscriptions.{DSL, Model}
import spray.routing.Route
import ru.yandex.tours.subscriptions.model.ModelConverters._
import scala.collection.JavaConverters._
import ru.yandex.tours.util.lang.Dates._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.09.15
 */
class SubscriptionsHandler extends HttpHandler with Bindings {

  private def toQuery(dates: SearchDates): Seq[Model.Query] = {
    val builder = Seq.newBuilder[Model.Query]

    builder += DSL.term(DSL.longRange("when", dates.when.start.toMillis, dates.when.end.toMillis))
    builder += DSL.term(DSL.points("nights", dates.nights.values.map(_.toString).asJava))
    dates.whenBack.foreach { whenBack =>
      DSL.term(DSL.longRange("when_back", whenBack.start.toMillis, whenBack.end.toMillis))
    }

    builder.result()
  }
  private def toQuery(filters: Seq[Filter]): Seq[Model.Query] = {
    val builder = Seq.newBuilder[Model.Query]
    for (filter <- filters if filter.nonEmpty) {
      builder += DSL.term(DSL.points(filter.name, filter.values.collect {
        case StringValue(value) => value
        case BooleanValue(value) => value.toString
        case IntValue(value) => value.toString
      }.asJava))
    }
    builder.result()
  }

  private def toQuery(params: Subscription.SubjectParams): Seq[Model.Query] = {
    val builder = Seq.newBuilder[Model.Query]

    builder += DSL.term(DSL.point("from", params.from.toString))
    builder += DSL.term(DSL.points("ages", params.ages.sorted.zipWithIndex.map { case (age, idx) => s"$idx:$age" }.asJava))
    builder += DSL.and(toQuery(params.dates).asJava)
    val filterQueries = toQuery(params.filters)
    if (filterQueries.nonEmpty) builder += DSL.and(filterQueries.asJava)

    builder.result()
  }

  override def route: Route = {
    (path("query") & metered("subscriptions.query")) {
      ctx =>
        val query = ctx.request.uri.query
        val subject = query.toSubject

        val modelQuery = subject match {
          case Direction(_, toId, params) =>
            val modelQueries = DSL.term(DSL.point("to", toId.toString)) +: toQuery(params)
            DSL.and(modelQueries: _*)
          case Hotel(_, hotelId, params) =>
            val modelQueries = DSL.term(DSL.point("hotel_id", hotelId.toString)) +: toQuery(params)
            DSL.and(modelQueries: _*)
        }

        completeProto(modelQuery)(ctx)
    }
  }
}
