package ru.yandex.tours.api.v1.users

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsIndex, HotelsVideo}
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.UserVisits
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.services.{MinPriceService, UserFavoritesService, UserHistoryService}
import ru.yandex.tours.util.spray.{HttpHandler, _}
import shapeless._
import spray.routing.{Directive1, Route}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 11.05.16
 */
class UsersHandler(routeesContext: RouteesContext,
                   hotelsIndex: HotelsIndex,
                   hotelsVideo: HotelsVideo,
                   tree: region.Tree,
                   geoMappingHolder: GeoMappingHolder,
                   searchSettingsHolder: SearchSettingsHolder,
                   userHistoryService: UserHistoryService,
                   userFavoritesService: UserFavoritesService)
  extends HttpHandler with Bindings with JsonSerialization {

  override val route: Route = pathPrefix(LongNumber) { uid ⇒
    pathPrefix("history") {
      getHistory(uid) ~
        pathPrefix(IntNumber) { hotelId ⇒
          putHistory(uid, hotelId) ~
          updateHistory(uid, hotelId)
        }
    } ~ pathPrefix("favorites") {
      getFavorites(uid) ~
        statsForFavorites(uid) ~
        checkFavorites(uid) ~
        pathPrefix(IntNumber) { hotelId ⇒
          putFavorites(uid, hotelId) ~
          deleteFavorites(uid, hotelId)
        }
    }
  }

  private case class PriceData(tourPrice: Option[Int], roomPrice: Option[Int])
  private case class RichRecord(record: UserVisits.Record, hotel: Hotel)
  private case class FavoritesFilter(countryIds: Set[Int], cityIds: Set[Int])

  private case class CountryStats(minPrice: Int, regions: Map[Region, Int]) {
    private def doUpdate(oldPrice: Int, newPrice: Int) = {
      oldPrice < 0 || (newPrice < oldPrice && newPrice >= 0)
    }
    def + (r: Region, newPrice: Int): CountryStats = {
      val newCountryPrice = if (doUpdate(minPrice, newPrice)) newPrice else minPrice
      val oldRegionPrice = regions.getOrElse(r, MinPriceService.UNKNOWN_PRICE)
      val newRegionsMap =
        if (doUpdate(oldRegionPrice, newPrice)) regions.updated(r, newPrice)
        else regions
      CountryStats(newCountryPrice, newRegionsMap)
    }
  }
  private case class FavoritesStats(countries: Map[Region, CountryStats]) {
    def + (record: RichRecord): FavoritesStats = {
      val RichRecord(r, hotel) = record
      val newPrice =
        if (r.getPrices.hasTourPrice) r.getPrices.getTourPrice
        else MinPriceService.UNKNOWN_PRICE

      (tree.country(hotel.geoId), tree.pathToRoot(hotel.geoId).find(isGoodRegion)) match {
        case (Some(country), Some(region)) ⇒
          val oldStat = countries.getOrElse(country, CountryStats(MinPriceService.UNKNOWN_PRICE, Map.empty))
          copy(countries.updated(country, oldStat + (region, newPrice)))
        case _ ⇒ this
      }
    }
  }

  private val favoritesFilter: Directive1[FavoritesFilter] = {
    (intArray("country_ids", isEmptyOk = true) & intArray("city_ids", isEmptyOk = true)) hmap {
      case countryIds :: cityIds :: HNil ⇒ FavoritesFilter(countryIds.toSet, cityIds.toSet)
    }
  }

  private val TOUR_PRICE = "tour_price"
  private val ROOM_PRICE = "room_price"
  private val FLIGHT_PRICE = "flight_price"

  private def visitItemToJson(record: UserVisits.Record, hotel: Hotel, lang: Lang) = {
    val obj = new JSONObject()
      .put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))
      .put("visit_time", record.getCreated)
      .put("record_id", record.getRecordId)

    if (record.hasRequest) {
      obj.put("search_request", toJson(HotelSearchRequest(record.getRequest), tree, lang, None))
    }

    val prices = record.getPrices
    val pricesObj = new JSONObject()
    if (prices.hasTourPrice) pricesObj.put(TOUR_PRICE, prices.getTourPrice)
    if (prices.hasRoomPrice) pricesObj.put(ROOM_PRICE, prices.getRoomPrice)
    if (prices.hasFlightPrice) pricesObj.put(FLIGHT_PRICE, prices.getFlightPrice)
    obj.put("prices", pricesObj)
    obj
  }

  private def visitItemsToArray(records: Seq[RichRecord], lang: Lang): JSONArray = {
    val json = new JSONArray()
    for (RichRecord(record, hotel) ← records) {
      json.put(visitItemToJson(record, hotel, lang))
    }
    json
  }

  private def enrichItems(records: Seq[UserVisits.Record]): Seq[RichRecord] = {
    val hotels = hotelsIndex.getHotelsById(records.map(_.getHotelId))
    for {
      record ← records
      hotelId = record.getHotelId
      hotel ← hotels.get(hotelId).orElse(hotelsIndex.getHotelById(hotelId))
    } yield RichRecord(record, hotel)
  }

  private def wrap(array: JSONArray): JSONObject = {
    new JSONObject()
      .put("list", array)
  }

  private def isGoodRegion(r: Region): Boolean = {
    geoMappingHolder.isKnownDestination(r.id) && searchSettingsHolder.getRegionSearchSettings(r).isSearchable
  }

  private def jsonToPriceData(json: JSONObject): UserVisits.PriceData = {
    val data = UserVisits.PriceData.newBuilder()
    if (json.has(TOUR_PRICE)) data.setTourPrice(json.getInt(TOUR_PRICE))
    if (json.has(ROOM_PRICE)) data.setRoomPrice(json.getInt(ROOM_PRICE))
    if (json.has(FLIGHT_PRICE)) data.setFlightPrice(json.getInt(FLIGHT_PRICE))
    data.build()
  }

  private def getHistory(uid: Long) = {
    (pathEndOrSingleSlash & get & parameter("limit".as[Int]) & lang) { (limit, lang) ⇒
      onSuccess(userHistoryService.getRecentItems(uid, limit)) { records ⇒
        val enriched = enrichItems(records)
        val json = visitItemsToArray(enriched, lang)
        completeJsonOk(wrap(json))
      }
    }
  }

  private def putHistory(uid: Long, hotelId: Int) = {
    withHotel(hotelsIndex, hotelId) { hotel ⇒
      (pathEndOrSingleSlash & put & routeesContext.offersSearchRequest(hotel) & entity(as[String])) { (request, body) ⇒
        val data = jsonToPriceData(new JSONObject(body))
        onSuccess(userHistoryService.putVisit(uid, hotel.id, request.hotelRequest, data)) { _ ⇒
          completeJsonOk(new JSONArray())
        }
      }
    }
  }

  private def updateHistory(uid: Long, hotelId: Int) = {
    withHotel(hotelsIndex, hotelId) { hotel ⇒
      (pathEndOrSingleSlash & post & routeesContext.offersSearchRequest(hotel) & entity(as[String])) { (request, body) ⇒
        val data = jsonToPriceData(new JSONObject(body))
        onSuccess(userHistoryService.updateVisit(uid, hotel.id, request.hotelRequest, data)) { _ ⇒
          completeJsonOk(new JSONArray())
        }
      }
    }
  }

  private def getFavorites(uid: Long) = {
    (pathEndOrSingleSlash & get & paging & favoritesFilter & lang) { (paging, filter, lang) ⇒
      onSuccess(userFavoritesService.getFavorites(uid)) { records ⇒
        val enriched = enrichItems(records)
        val filtered = enriched.filter {
          case RichRecord(_, hotel) ⇒
            val path = tree.pathToRoot(hotel.geoId).map(_.id)
            val countryPass = filter.countryIds.isEmpty || path.exists(filter.countryIds.contains)
            val cityPass = filter.cityIds.isEmpty || path.exists(filter.cityIds.contains)
            countryPass && cityPass
        }
        val sliced = paging(filtered)
        completeJsonOk(toJson(visitItemsToArray(sliced, lang), paging, records.size))
      }
    }
  }

  private def statsForFavorites(uid: Long) = {
    (path("statistics") & get & lang) { lang ⇒
      onSuccess(userFavoritesService.getFavorites(uid)) { records ⇒
        val enriched = enrichItems(records)

        val stats = enriched.foldLeft(FavoritesStats(Map.empty)) { _ + _ }
        log.info("stats: " + stats)
        val res = new JSONArray()
        for ((country, stat) ← stats.countries) {
          val obj = new JSONObject()
          obj.put("region", regionToJson(country, lang))
          if (stat.minPrice > 0) obj.put("min_price", stat.minPrice)
          for ((city, minPrice) ← stat.regions) {
            val cityObj = new JSONObject()
            cityObj.put("region", regionToJson(city, lang))
            if (minPrice > 0) cityObj.put("min_price", minPrice)
            obj.append("regions", cityObj)
          }
          res.put(obj)
        }
        completeJsonOk(res)
      }
    }
  }

  private def checkFavorites(uid: Long) = {
    (path("check") & get & intArray("hotel_ids")) { hotelIds ⇒
      onSuccess(userFavoritesService.getFavorites(uid, hotelIds)) { records ⇒
        val favoriteHotels = records.map(_.getHotelId).toSet

        val res = new JSONObject()
        for (id ← hotelIds) {
          res.put(id.toString, favoriteHotels.contains(id))
        }
        completeJsonOk(res)
      }
    }
  }

  private def putFavorites(uid: Long, hotelId: Int): Route = {
    (pathEndOrSingleSlash & put & entity(as[String])) { body ⇒
      withHotel(hotelsIndex, hotelId) { hotel ⇒
        routeesContext.offersSearchRequest(hotel) { request ⇒
          val data = jsonToPriceData(new JSONObject(body))
          onSuccess(userFavoritesService.putVisit(uid, hotel.id, request.hotelRequest, data)) { _ ⇒
            completeJsonOk(new JSONArray())
          }
        }
      }
    }
  }

  private def deleteFavorites(uid: Long, hotelId: Int): Route = {
    (pathEndOrSingleSlash & delete) {
      onSuccess(userFavoritesService.deleteFavorites(uid, hotelId)) { _ ⇒
        completeJsonOk(new JSONArray())
      }
    }
  }
}
