package ru.yandex.tours.api.v1.view

import org.json.JSONObject
import ru.yandex.tours.api.JsonSerialization._
import ru.yandex.tours.avia.{Airport, Airports, FlightUtils}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.model.search.SearchProducts.FlightTrip
import ru.yandex.tours.model.search.SearchResults.FlightSearchResult
import spray.http.Uri

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 15.02.16
 */
class FlightSearchResultView(request: FlightSearchRequest,
                             flightAlternatives: FlightAlternativesView,
                             flights: FlightSearchResult) {

  private val adults = request.hotelRequest.ages.count(_ >= 12)
  private val children = request.hotelRequest.ages.count(a => a < 12 && a >= 2)
  private val infants = request.hotelRequest.ages.count(_ < 2)
  private val aviaUrl = Uri(s"https://avia.yandex.ru/search/").withQuery(
    "fromId" -> ("g" + request.hotelRequest.from),
    "toId" -> request.airportId,
    "when" -> request.hotelRequest.when.toString,
    "return_date" -> request.hotelRequest.whenBack.toString,
    "adult_seats" -> adults.toString,
    "children_seats" -> children.toString,
    "infant_seats" -> infants.toString,
    "klass" -> "economy",
    "utm_source" -> "yandex_travel"
  )

  def getTripDuration(trip: FlightTrip): Long = {
    val forward = FlightUtils.getFlightDuration(trip.getForwardList.asScala)
    val backward = FlightUtils.getFlightDuration(trip.getBackwardList.asScala)
    forward.getStandardMinutes + backward.getStandardMinutes
  }

  def asJson(airports: Airports, tree: region.Tree, lang: Lang): JSONObject = {
    def airportToJson(airport: Airport): JSONObject = {
      val res = new JSONObject()
        .put("airport_id", airport.aviaId)
        .put("name", airport.name(lang))
      airport.geoId.flatMap(tree.region).foreach { region =>
        res.put("region", regionToJson(region, lang))
      }
      res
    }
    def flightToJson(flight: FlightTrip.Flight) = {
      val json = new JSONObject()
      json.put("flight_number", flight.getFlightNumber)
      json.put("duration", flight.getDuration)
      json.put("company", flight.getCompany)
      for (departure <- airports.byAviaId(flight.getDeparture)) json.put("departure_airport", airportToJson(departure))
      for (arrival <- airports.byAviaId(flight.getArrival)) json.put("arrival_airport", airportToJson(arrival))
      json.put("departure_time", flight.getDepartureTime)
      json.put("arrival_time", flight.getArrivalTime)
      json
    }
    def tripToJson(trip: FlightTrip): JSONObject = {
      new JSONObject()
        .put("min_price", trip.getMinPrice)
        .put("forward", trip.getForwardList.asScala.map(flightToJson).asJavaCollection)
        .put("backward", trip.getBackwardList.asScala.map(flightToJson).asJavaCollection)
        .put("url", aviaUrl.toString)
    }

    val res = new JSONObject()
    res.put("search_request", {
      toJson(request.hotelRequest, tree, lang, None)
        .put("airport_id", request.airportId)
        .put("airport", airports.byId(request.airportId).map(airportToJson).orNull)
    })
    res.put("result_info", toJson(flights.getResultInfo, tooCloseDestination = flightAlternatives.isEmpty))
    res.put("progress", toJson(flights.getProgress))
    res.put("url", aviaUrl)

    res.put("alternatives", flightAlternatives.asJson(tree, lang))

    val flightsJson = new JSONObject()

    if (flights.getFlightTripsCount > 0) {
      val cheapest = flights.getFlightTripsList.asScala.minBy(f => (f.getMinPrice, getTripDuration(f)))
      val fastest = flights.getFlightTripsList.asScala.minBy(f => (getTripDuration(f), f.getMinPrice))

      flightsJson.put("cheapest", tripToJson(cheapest))
      if (fastest != cheapest) {
        flightsJson.put("fastest", tripToJson(fastest))
      }
    }

    res.put("flights", flightsJson)

    res
  }
}
