package ru.yandex.tours.api.v1.view

import com.wordnik.swagger.annotations.{ApiModel, ApiModelProperty}
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization._
import ru.yandex.tours.backend.HotelSnippetPreparer.HotelsWithInfo
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsVideo
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.filter.Filter
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchResults.ErrorCode
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.model.{MapRectangle, Source}
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.http.UrlLabelerBuilder

import scala.annotation.meta.field

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 01.02.16
 */
@ApiModel(value = "Results of search request")
case class SearchResultView(@(ApiModelProperty@field)(value = "Search request", dataType = "json")
                            searchRequest: HotelSearchRequest,
                            @(ApiModelProperty@field)(value = "Filters", dataType = "array[json]")
                            filters: Seq[Filter],
                            @(ApiModelProperty@field)(value = "Sorting", dataType = "json")
                            sortBy: SortType,
                            @(ApiModelProperty@field)(value = "language", dataType = "string")
                            lang: Lang,
                            @(ApiModelProperty@field)(value = "requested page and page size", dataType = "json")
                            paging: Paging,
                            @(ApiModelProperty@field)(value = "result snippets", dataType = "json")
                            snippetsWithInfo: HotelsWithInfo) {

  def asJson[T <: Source](tree: Tree, searchSources: SearchSources[T], hotelsVideo: HotelsVideo,
                          geoMapping: GeoMappingHolder, searchSettings: SearchSettingsHolder,
                          labelerBuilder: UrlLabelerBuilder): JSONObject = {
    val json = new JSONObject()

    val statistic = snippetsWithInfo.statistic
    val resultInfo = snippetsWithInfo.resultInfo
    val progressJson = toJson(snippetsWithInfo.progress, statistic.found, statistic.total)
      .put("tours", statistic.offerStatistic.found)
      .put("total_tours", statistic.offerStatistic.total)
    json.put("progress", progressJson)
    json.put("statistic", toJson(statistic.offerStatistic))
    json.put("map", toJson(getMapInfo(searchRequest.to, tree, snippetsWithInfo)))
    json.put("search_request", toJson(searchRequest, tree, lang, None))
    val allFiltered = statistic.found == 0 && statistic.total != 0
    val regionIfCountry = snippetsWithInfo.resultInfo.hasError &&
      snippetsWithInfo.resultInfo.getError == ErrorCode.TOO_LARGE_DESTINATION
    val resultInfoJs = toJson(resultInfo, regionIfCountry, filters.exists(_.nonEmpty), allFiltered)
    json.put("result_info", resultInfoJs)
    // TODO remove after front release
    json.put("country_search", regionIfCountry)
    val jsonFilters = filtersToJson(filters)
    jsonFilters.put("sort", sortBy.toString.toLowerCase)
    json.put("filters", jsonFilters)
    val allowedState = new JSONObject()
    snippetsWithInfo.allowedStates.foreach(_.putToJson(allowedState))
    json.put("allowed_filters", allowedState)
    val array = new JSONArray()
    for (snippet <- snippetsWithInfo.snippets) {
      val snippetJson = snippetToJson(snippet, searchRequest.ages, tree, searchSources,
        hotelsVideo, geoMapping, searchSettings, lang, labelerBuilder.buildFrom(snippet.sample))
      array.put(snippetJson)
    }
    json.put("tours", toJson(array, paging, statistic.found))

    json
  }

  private def getMapInfo(to: Int, tree: Tree, result: HotelsWithInfo) = {
    if (result.snippets.isEmpty) {
      tree.region(to).map(_.boundingBox).getOrElse(MapRectangle.empty)
    } else {
      MapRectangle.byPoints(result.snippets.map(s => (s.snippet.hotel.longitude, s.snippet.hotel.latitude)))
    }
  }
}
