package ru.yandex.tours.api.v1.view

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.api.JsonSerialization._
import ru.yandex.tours.avia.Airports
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.search.SearchProducts.TransferOption
import ru.yandex.tours.model.search.SearchResults.TransferSearchResult
import ru.yandex.tours.model.search.TransferSearchRequest

import scala.collection.JavaConversions._

/**
  * Created by asoboll on 28.04.16.
  */
class TransferSearchResultView(request: TransferSearchRequest, transfers: TransferSearchResult) {

  private def MAX_CHEAPEST_OPTIONS_COUNT = 3

  private def transferToJson(transfer: TransferOption) = {
    val json = new JSONObject()
      .put("type", transfer.getType)
      .put("unified_type", if (transfer.hasUnifiedType) transfer.getUnifiedType else null)
      .put("name", transfer.getName)
    if (transfer.hasNameDetailed) json.put("name_detailed", transfer.getNameDetailed)
    json.put("pickup_location", transfer.getPickupLoc)
      .put("pickup_time", transfer.getPickupTime)
      .put("dropoff_time", transfer.getDropoffTime)
      .put("image_url", transfer.getImage)
    if (transfer.hasDropoffLoc) json.put("dropoff_location", transfer.getDropoffLoc)
    if (transfer.hasImageVendor) json.put("image_vendor_url", transfer.getImageVendor)
    json.put("price", transfer.getPrice)
    if (transfer.hasPriceDaily) json.put("price_daily", transfer.getPriceDaily)
    json.put("link", transfer.getLink)
    val details = new JSONObject()
    for (detail <- transfer.getDetailsList) {
      details.put(detail.getName, detail.getValue)
    }
    json.put("details", details)
  }

  def asJson(tree: region.Tree, lang: Lang, onlyCheapest: Boolean = true, airports: Airports): JSONObject = {
    def requestJson = toJson(request.hotelRequest, tree, lang, None)
      .put("airport_id", request.airportId)
      .put("airport_name", airports.byId(request.airportId).map(_.name(lang)).orNull)
      .put("hotel_id", request.hotelId)
      .put("request_url", if (transfers.hasRequestUrl) transfers.getRequestUrl else null)

    def baseJson = new JSONObject()
      .put("search_request", requestJson)
      .put("result_info", toJson(transfers.getResultInfo, tooCloseDestination = false))
      .put("progress", toJson(transfers.getProgress))

    val transferOptions: Seq[TransferOption] =
      if (onlyCheapest) transfers.getTransferOptionList.groupBy(_.getUnifiedType).values.map(_.minBy(_.getPrice)).toSeq
      else transfers.getTransferOptionList

    val res = baseJson
    val transfersJson = new JSONArray()
    for (transfer <- transferOptions.sortBy(_.getPrice).take(MAX_CHEAPEST_OPTIONS_COUNT))
      transfersJson.put(transferToJson(transfer))
    res.put("transfers", transfersJson)

    res
  }
}
