package ru.yandex.tours.backend

import ru.yandex.tours.backend.HotelSnippetPreparer._
import ru.yandex.tours.billing.BillingOffers.IdedOfferBilling
import ru.yandex.tours.billing.{BillingIndex, BillingService, CMBillingOffers}
import ru.yandex.tours.filter.SnippetFiltrator
import ru.yandex.tours.filter.State.AllowedState
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.hotels.{HotelRatings, HotelsIndex, HotelsSimilarity}
import ru.yandex.tours.model.filter.hotel.StarFilter
import ru.yandex.tours.model.filter.{Filter, SnippetFilter}
import ru.yandex.tours.model.hotels.{Hotel, Partners, RichHotelSnippet}
import ru.yandex.tours.model.purchase.BillingObject
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchResults.{ErrorCode, ResultInfo, SearchProgress}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest, SearchFilter, SearchType}
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.model.util.{Paging, SortType}
import ru.yandex.tours.model.{MapRectangle, Source}
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.search.settings.{RegionSearchSettings, SearchSettingsHolder}
import ru.yandex.tours.services.HotelSearchService
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.hotels.HotelSnippetUtil
import ru.yandex.vertis.billing.model.Properties

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Random

object HotelSnippetPreparer {
  case class OfferWithBilling(offer: Offer, billingObject: Option[BillingObject])

  case class SnippetWithBilling(snippet: RichHotelSnippet, offerBilling: Option[BillingObject], hotelBilling: Option[IdedOfferBilling]) {
    def sample: Offer = snippet.snippet.getSample
  }

  case class HotelsWithInfo(snippets: Iterable[SnippetWithBilling], progress: SearchProgress, statistic: HotelStatistic, resultInfo: ResultInfo, allowedStates: Iterable[AllowedState])

  case class HotelContextualMinPrice(isFinished: Boolean, continue: Boolean, minPrice: Option[Int])

  case class OffersWithInfo(tours: Iterable[OfferWithBilling], progress: SearchProgress, statistic: OfferStatistic, allowedStates: Iterable[AllowedState], resultInfo: ResultInfo)

  case class HotelStatistic(found: Int, total: Int, offerStatistic: OfferStatistic)

  case class OfferStatistic(found: Int, total: Int, minPrice: Option[Int]) {
    def allFiltered: Boolean = found == 0 && total > 0
  }
}

class HotelSnippetPreparer[T <: Source](searcher: HotelSearchService,
                                        hotelsIndex: HotelsIndex,
                                        hotelsSimilarity: HotelsSimilarity,
                                        tree: region.Tree,
                                        snippetFiltrator: SnippetFiltrator,
                                        hotelRatings: HotelRatings,
                                        billingService: BillingService,
                                        billingIndex: BillingIndex,
                                        cmBillingIndex: BillingIndex,
                                        searchSources: SearchSources[T],
                                        protected val searchType: SearchType,
                                        protected val searchSettings: SearchSettingsHolder)
                                       (implicit ec: ExecutionContext) {

  def getHotelSnippets(searchRequest: HotelSearchRequest,
                       paging: Paging,
                       sort: SortType,
                       filters: Iterable[Filter],
                       canStartRequest: Boolean,
                       userIdentifiers: UserIdentifiers): Future[HotelsWithInfo] = {
    val request = searchRequest.copy(filter = SearchFilter(filters, searchType))
    for (result <- searcher.search(request, canStartRequest, onlyFromLongCache = false)) yield {
      val rawSnippets = result.getHotelSnippetList
      val filteredSnippets = snippetFiltrator.filter(searchRequest, searchType, rawSnippets, filters)
      val mergedSnippets = mergeSnippets(filteredSnippets)
      val (randomPaid, rest) = randomPaidSplit(mergedSnippets, 1, userIdentifiers)
      val sortedSnippets = sortSnippets(rest, sort)
      val pagedResult = paging(randomPaid ++ sortedSnippets)
      val statistic = HotelStatistic(
        mergedSnippets.map(_.getHotelId).toSet.size,
        rawSnippets.map(_.getHotelId).toSet.size,
        OfferStatistic(
          mergedSnippets.map(_.getOfferCount).sum,
          rawSnippets.map(_.getOfferCount).sum,
          filteredSnippets.map(_.getPriceMin).minOpt
        )
      )
      val enriched = HotelSnippetUtil.enrich(pagedResult, hotelsIndex).map { s =>
        SnippetWithBilling(s, getOfferBilling(s.snippet), getHotelBilling(s.hotel))
      }
      HotelsWithInfo(
        enriched, result.getProgress, statistic, result.getResultInfo,
        snippetFiltrator.getAllowedState(searchRequest, rawSnippets, filters, searchType)
      )
    }
  }

  def getSimilarHotels(searchRequest: HotelSearchRequest,
                       hotel: Hotel,
                       count: Int): Future[Seq[(Hotel, Option[HotelSnippet], Option[BillingObject], Option[IdedOfferBilling])]] = {
    for (result <- searcher.search(searchRequest, canStartRequest = false, onlyFromLongCache = true)) yield {
      val snippets = result.getHotelSnippetList.map(s => s.getHotelId -> (s, getOfferBilling(s))).toMap
      val similar = hotelsSimilarity.getSimilar(hotel.id)
      val (sameCity, sameCountry) = similar.partition(_.geoId == hotel.geoId)
      val near = hotelsIndex.near(hotel, count, new StarFilter(Set(hotel.star.id)))

      val similar2 = (sameCity ++ sameCountry ++ near).distinct.map(h => h -> snippets.get(h.id))
      val withSnippet = similar2.filter(_._2.isDefined)
      val toShow =
        if (withSnippet.size >= count) withSnippet.take(count)
        else if (similar2.size >= count) similar2.take(count)
        else Nil
      toShow.map {
        case (h, Some((snippet, billing))) => (h, Some(snippet), billing, getHotelBilling(h))
        case (h, None) => (h, None, None, getHotelBilling(h))
      }
    }
  }

  def getFromLongCache(request: HotelSearchRequest, size: Int): Future[Seq[SnippetWithBilling]] = {
    for {
      result <- searcher.search(request, canStartRequest = false, onlyFromLongCache = true)
      snippets = mergeSnippets(result.getHotelSnippetList)
      sorted = sortSnippets(snippets.toSeq, SortType.RELEVANCE)
    } yield HotelSnippetUtil.enrich(sorted.take(size), hotelsIndex).map { s =>
      SnippetWithBilling(s, getOfferBilling(s.snippet), getHotelBilling(s.hotel))
    }
  }

  def getMapInfo(searchRequest: HotelSearchRequest,
                 filters: Iterable[Filter],
                 mapInfo: MapRectangle): Future[Iterable[Hotel]] = {
    for (result <- searcher.search(searchRequest, canStartRequest = false, onlyFromLongCache = false)) yield {
      val filtered = snippetFiltrator.filter(searchRequest, searchType, result.getHotelSnippetList, filters)
      val hotelIds = filtered.map(_.getHotelId).toSet
      val hotels = hotelsIndex.getHotelsById(hotelIds).values
      HotelsIndex.inRectangle(mapInfo, hotels)
    }
  }

  def getOffersInHotel(request: OfferSearchRequest,
                       sortType: SortType,
                       paging: Paging,
                       filters: Iterable[SnippetFilter],
                       canStartRequest: Boolean): Future[OffersWithInfo] = {
    for (result <- searcher.searchHotel(request, canStartRequest)) yield {
      val filtered = result.getOfferList.filter(offer => filters.forall(_.fits(offer)))
      val sorted = sortOffers(filtered, sortType)
      val paged = paging(sorted).map(offer => OfferWithBilling(offer, getBillingObject(offer)))
      val statistic = OfferStatistic(
        filtered.size,
        result.getOfferList.size,
        filtered.map(_.getPrice).minOpt
      )
      val allowedState = snippetFiltrator.getAllowedStateForOffers(result.getOfferList, filters)
      OffersWithInfo(paged, result.getProgress, statistic, allowedState, result.getResultInfo)
    }
  }

  def getHotelMinPrice(request: HotelSearchRequest, hotel: Hotel): Future[HotelContextualMinPrice] = {
    if (!hotel.searchAvailable(searchType)) {
      return Future.successful(HotelContextualMinPrice(isFinished = true, continue = false, None))
    }

    val offerSearchRequest = OfferSearchRequest(request, hotel.id)

    def searchInHotel = searcher.searchHotel(offerSearchRequest, canStartRequest = false).map { result =>
      val offers = result.getOfferList.asScala
      val continue = cacheMiss(result.getResultInfo)
      HotelContextualMinPrice(result.getProgress.getIsFinished, continue, getOfferMinPrice(offers))
    }

    def searchInRegion = searcher.search(request, canStartRequest = false, onlyFromLongCache = false).map {
      result =>
        val allSnippets = result.getHotelSnippetList.asScala
        val snippets = allSnippets.filter(_.getHotelId == hotel.id)
        val isFinished = result.getProgress.getIsFinished
        val minPrice = getSnippetMinPrice(snippets)
        val hotelOutsideSearchRegion = tree.pathToRoot(hotel.geoId).forall(_.id != request.to)
        //continue on cache miss or cache hit with non-empty result or hotel is outside search region
        val continue = cacheMiss(result.getResultInfo) || (allSnippets.nonEmpty && minPrice.isEmpty) ||
          hotelOutsideSearchRegion
        HotelContextualMinPrice(isFinished, continue, minPrice)
    }

    def startSearch = searcher.searchHotel(offerSearchRequest, canStartRequest = true).map { result =>
      val offers = result.getOfferList.asScala
      HotelContextualMinPrice(result.getProgress.getIsFinished, continue = false, getOfferMinPrice(offers))
    }

    val step0 = Future.successful(HotelContextualMinPrice(isFinished = false, continue = true, minPrice = None))
    val pipeline = Seq(() => searchInRegion, () => searchInHotel, () => startSearch)

    pipeline.foldLeft(step0) {
      case (future, nextStep) => future.flatMap {
        case result if !result.continue => Future.successful(result)
        case _ => nextStep()
      }
    }
  }

  def getHotelsMinPrice(request: HotelSearchRequest,
                        hotels: Seq[Hotel]): Future[Map[Hotel, HotelContextualMinPrice]] = {
    searcher.search(request, canStartRequest = false, onlyFromLongCache = false).map { result =>
      val useful = cacheHit(result.getResultInfo)
      if (useful) {
        val snippets = result.getHotelSnippetList.asScala.map(s => s.getHotelId -> s).toMultiMap
        hotels.map { hotel =>
          val hotelSnippets = snippets.getOrElse(hotel.id, Seq.empty)
          val isFinished = !hotel.searchAvailable(searchType) || {
            result.getProgress.getIsFinished && (hotelSnippets.nonEmpty || snippets.isEmpty)
          }
          val minPrice = getSnippetMinPrice(hotelSnippets)
          val continue = hotel.searchAvailable(searchType) && result.getProgress.getIsFinished &&
            minPrice.isEmpty && snippets.nonEmpty
          hotel -> HotelContextualMinPrice(isFinished, continue, minPrice)
        }.toMap
      } else {
        hotels.map { hotel =>
          hotel -> HotelContextualMinPrice(isFinished = !hotel.searchAvailable(searchType),
            continue = hotel.searchAvailable(searchType), minPrice = None)
        }.toMap
      }
    }.flatMap { result =>
      val hotelsToLoad = result.collect {
        case (hotel, mp) if mp.continue => hotel
      }.toSeq

      if (hotelsToLoad.nonEmpty) {
        val hotelsMap = hotelsToLoad.map(h => h.id -> h).toMap
        val futures = hotelsToLoad.map(_.id).grouped(100).map { hotelIds =>
          searcher.searchHotels(request, hotelIds, canStartRequest = false)
        }
        Future.fold(futures)(result) { (prevResult, searchResults) =>
          val updated = for {
            hotelResult <- searchResults
            hotel <- hotelsMap.get(hotelResult.getHotelId)
          } yield {
            val offers = hotelResult.getOfferList.asScala
            val isFinished = hotelResult.getProgress.getIsFinished && cacheHit(hotelResult.getResultInfo)
            hotel -> HotelContextualMinPrice(isFinished, continue = false, getOfferMinPrice(offers))
          }
          prevResult ++ updated
        }
      } else {
        Future.successful(result)
      }
    }
  }

  private def mergeSnippets(snippets: Iterable[HotelSnippet]) = {
    snippets.groupBy(_.getHotelId).map { case (_, same) => same.reduce(HotelSnippetUtil.merge) }
  }

  private def sortSnippets(snippets: Seq[HotelSnippet], sort: SortType): Seq[HotelSnippet] = {
    sort match {
      case SortType.PRICE_ASC => snippets.sortBy(_.getPriceMin)
      case SortType.RATING_DESC => sortSnippets(snippets, hotelId ⇒ -hotelRatings.getRating(hotelId))
      case SortType.PRICE_DESC => snippets.sortBy(-_.getPriceMin)
      case SortType.RATING_ASC => sortSnippets(snippets, hotelId ⇒ hotelRatings.getRating(hotelId))
      case SortType.RELEVANCE =>
        val result = sortSnippets(snippets, hotelId ⇒ -hotelRatings.getRelevance(hotelId))
        if (result.size > 3) {
          val snippet = result.minBy(_.getPriceMin)
          val filtered = result.filterNot(_ == snippet)
          filtered.take(2) ++ Seq(snippet) ++ filtered.drop(2)
        } else {
          result
        }
      case SortType.VISITS =>
        val result = sortSnippets(snippets, hotelId ⇒ -hotelRatings.getVisits(hotelId))
        if (result.size > 3) {
          val snippet = result.minBy(_.getPriceMin)
          val filtered = result.filterNot(_ == snippet)
          filtered.take(2) ++ Seq(snippet) ++ filtered.drop(2)
        } else {
          result
        }
    }
  }

  private def sortSnippets(snippets: Seq[HotelSnippet], f: Int => Double): Seq[HotelSnippet] = {
    snippets.map(s => (s, f(s.getHotelId))).sortBy(p => (p._2, p._1.getPriceMin)).map(_._1)
  }

  private def sourcePriority(offer: Offer): Double = {
    searchSources.getById(offer.getSource.getOperatorId).fold(30d)(_.priority)
  }

  private def sortOffers(offers: Seq[Offer], sortType: SortType): Seq[Offer] = {

    sortType match {
      case SortType.PRICE_DESC ⇒ offers.sortBy(o ⇒ (-o.getPrice, -sourcePriority(o)))
      case _ ⇒ offers.sortBy(o ⇒ (o.getPrice, -sourcePriority(o)))
    }
  }

  private def cacheMiss(resultInfo: ResultInfo) = resultInfo.hasError && resultInfo.getError == ErrorCode.NOT_STARTED
  private def cacheHit(resultInfo: ResultInfo) = !cacheMiss(resultInfo)
  private def getOfferMinPrice(offers: Seq[Offer]) = offers.map(_.getPrice).minOpt
  private def getSnippetMinPrice(snippets: Seq[HotelSnippet]) = snippets.map(_.getPriceMin).minOpt

  // User identifiers is needed here for stable random per user
  private def randomPaidSplit(snippets: Iterable[HotelSnippet],
                              size: Int,
                              userIdentifiers: UserIdentifiers): (Seq[HotelSnippet], Seq[HotelSnippet]) = {
    if (searchType == SearchType.ROOMS) {
      val seed = userIdentifiers.hashCode()
      val random = new Random(seed)
      val paid = snippets.filter(s => getHotelBilling(s.getHotelId).isDefined)
      val randomPaid = random.shuffle(paid).take(size)
      val paidIds = randomPaid.map(_.getHotelId).toSet
      (randomPaid.toSeq, snippets.filter(s => !paidIds.contains(s.getHotelId)).toSeq)
    } else {
      (Seq.empty, snippets.toSeq)
    }
  }

  private def cmHotelBillingObject(offer: Offer): Option[BillingObject] = {
    for {
      idedOfferBilling <- cmBillingIndex.getOfferBilling(offer.getHotelId)
      billingObject <- CMBillingOffers.toBillingObject(idedOfferBilling).toOption
    } yield billingObject
  }

  private def hotelBillingObject(offer: Offer): Option[BillingObject] = {
    for {
      partner <- Partners.getOpt(offer.getSource.getPartnerId)
      operator <- searchSources.getById(offer.getSource.getOperatorId)
      billingInfo <- Partners.isChannelManager(partner) match {
        case false => billingService.getBillingInfo(operator.code, offer.getId)
        case true => cmHotelBillingObject(offer)
      }
    } yield billingInfo
  }

  private def getBillingObject(offer: Offer): Option[BillingObject] = {
    searchType match {
      case SearchType.ROOMS =>
        for {
          billingInfo <- hotelBillingObject(offer)
        } yield {
          billingInfo.clickRevenue match {
            case Some(revenue) => billingInfo.addUiInfo(Properties.BILLING_CLICK_REVENUE -> revenue.toString)
            case None => billingInfo
          }
        }
      case _ => None
    }
  }

  private def getOfferBilling(snippet: HotelSnippet): Option[BillingObject] =
    if (snippet.hasSample) getBillingObject(snippet.getSample) else None

  private def getHotelBilling(hotel: Hotel) = billingIndex.getOfferBilling(hotel.id)

  private def getHotelBilling(id: Int) = billingIndex.getOfferBilling(id)
}

trait SearchSettingsSupport[T <: Source] extends HotelSnippetPreparer[T] {
  import SearchSettingsSupport._

  override def getHotelSnippets(searchRequest: HotelSearchRequest,
                                paging: Paging,
                                sort: SortType,
                                filters: Iterable[Filter],
                                canStartRequest: Boolean,
                                userIdentifiers: UserIdentifiers): Future[HotelsWithInfo] = {
    val settings: RegionSearchSettings = searchSettings.getRegionSearchSettings(searchRequest.to)

    if (settings.banned(searchType)) {
      Future.successful(resultForBanned)
    } else if (settings.disallowed(searchType)) {
      Future.successful(resultForDisallowed)
    } else super.getHotelSnippets(searchRequest, paging, sort, filters, canStartRequest, userIdentifiers)
  }
}

object SearchSettingsSupport {
  private val emptyFinishedProgress = SearchProgress.newBuilder()
    .setOperatorCompleteCount(0)
    .setOperatorTotalCount(0)
    .setOperatorFailedCount(0)
    .setOperatorSkippedCount(0).
    setIsFinished(true)
    .build()

  private val emptyStatistic = HotelStatistic(0, 0, OfferStatistic(0, 0, None))

  private def resultInfo(errorCode: ErrorCode) =
    ResultInfo.newBuilder().
      setIsFromLongCache(false).
      setError(errorCode).
      build()

  val resultForBanned = HotelsWithInfo(
    Iterable.empty,
    emptyFinishedProgress,
    emptyStatistic,
    resultInfo(ErrorCode.SUSPENDED_DESTINATION),
    Iterable.empty
  )

  val resultForDisallowed = HotelsWithInfo(
    Iterable.empty,
    emptyFinishedProgress,
    emptyStatistic,
    resultInfo(ErrorCode.TOO_LARGE_DESTINATION),
    Iterable.empty
  )
}

