package ru.yandex.tours.backend

import play.api.libs.json._
import ru.yandex.tours.billing.BillingService
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.ota.OnlineTourAgency
import ru.yandex.tours.model.purchase.{PurchaseObject, PurchaseWithGet, PurchaseWithPost}
import ru.yandex.tours.model.search.OfferSearchRequest
import ru.yandex.tours.model.search.SearchProducts.Actualization._
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.ota.{AuctionService, OnlineTourAgencies, OtaMappingHolder}
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.parsing.IntValue
import ru.yandex.tours.util.{Encryption, Logging}

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.util.Try
import scala.util.control.NonFatal

class PurchaseService(billingService: BillingService,
                      otaMapping: OtaMappingHolder,
                      geoMappingHolder: GeoMappingHolder,
                      hotelsIndex: HotelsIndex,
                      tree: Tree,
                      tourOperators: TourOperators,
                      agencies: OnlineTourAgencies,
                      auctionService: AuctionService) extends Logging {

  private def getStaticLinks(actualizedOffer: ActualizedOffer): Seq[PurchaseObject] = {
    val tourOperatorCodes = tourOperators.getAll.map(_.code).toSet
    val offer = actualizedOffer.getOffer
    val linkList =
      if (actualizedOffer.hasActualizationInfo && (actualizedOffer.getActualizationInfo.getLinkOverrideCount > 0)) {
        actualizedOffer.getActualizationInfo.getLinkOverrideList
      } else {
        offer.getLinkList
      }

    for {
      link ← linkList.asScala
      billingInfo ← billingService.getBillingInfo(link.getBillingId, offer.getId)
      obj ← Try {
        PurchaseWithGet(
          link = link.getLink,
          partnerName = link.getPartnerName,
          billingId = link.getBillingId,
          billing = billingInfo,
          isTourOperator = tourOperatorCodes.contains(link.getBillingId)
        )
      }.onFailure {
        case NonFatal(e) ⇒ log.error("Can not create static purchase link for offer!", e)
      }.toOption
    } yield obj
  }

  def getPurchaseLinks(request: OfferSearchRequest,
                       actualizedOffer: ActualizedOffer,
                       utm: UtmMark,
                       userIdentifiers: UserIdentifiers,
                       visibleButtonCount: Int): Seq[PurchaseObject] = {
    if (actualizedOffer.getResultInfo.getIsFromLongCache) {
      Seq.empty
    } else {
      val links = Seq(
        getStaticLinks(actualizedOffer),
        agencies.agencies.flatMap(agency => getPostPurchase(request, actualizedOffer, agency, utm, userIdentifiers))
      ).flatten
      auctionService.handle(links, visibleButtonCount - 1)
    }
  }

  private def getPostPurchase(json: JsObject, agency: OnlineTourAgency, offerId: String): Option[PurchaseWithPost] = {
    val offer = json.toString()
    val signature = getSignature(offer, agency)
    billingService.getBillingInfo(agency.billingId, offerId).map(billing =>
      PurchaseWithPost(agency.actionUrl, offer, signature, agency.name, agency.billingId, billing)
    )
  }

  private def getPostPurchase(request: OfferSearchRequest,
                              actualizedOffer: ActualizedOffer,
                              agency: OnlineTourAgency,
                              utm: UtmMark,
                              userIdentifiers: UserIdentifiers): Option[PurchaseWithPost] = {
    generateActualizedJson(request, actualizedOffer, agency).flatMap(json => {
      getPostPurchase(json, agency, actualizedOffer.getOffer.getId)
    })
  }

  private def generateOfferJson(request: OfferSearchRequest,
                                actualizedOffer: ActualizedOffer,
                                agency: OnlineTourAgency): Option[JsObject] = {
    val offer = actualizedOffer.getOffer
    for {
      operator ← tourOperators.getById(offer.getSource.getOperatorId)
      hotel ← hotelsIndex.getHotelById(offer.getHotelId)
      geoBaseCountry ← tree.country(hotel.geoId)
      if otaMapping.isKnownOperator(agency, operator)
      if otaMapping.isKnownHotel(agency, hotel.id)
      if otaMapping.isKnownCity(agency, request.hotelRequest.from)
    } yield {
      val ages = request.hotelRequest.getAges

      val bookingUrl =
        if (offer.hasAgentBookingUrl) Seq("booking_url" -> JsString(offer.getAgentBookingUrl))
        else Seq.empty

      // Now we are using LT ids for regions, because they are public for OTAs
      // Also, here we use knowledge, that LT ids is Ints
      val optFields = Seq(
        toIntJs("hotel", hotel.partnerId(Partners.lt).headOption),
        toIntJs("operator", operator.mappingFor(Partners.lt).headOption),
        toIntJs("resort", geoMappingHolder.getPartnerCity(Partners.lt, hotel.geoId)),
        toIntJs("country", geoMappingHolder.getPartnerCountry(Partners.lt, geoBaseCountry.id)),
        toIntJs("departure_city", geoMappingHolder.getPartnerDeparture(Partners.lt, request.hotelRequest.from)),
        bookingUrl
      ).flatten

      val price = {
        if (actualizedOffer.hasActualizationInfo) actualizedOffer.getActualizationInfo.getPrice
        else offer.getPrice
      }

      Json.obj(
        "has_transfer" -> offer.getWithTransfer,
        "has_medical_insurance" -> offer.getWithMedicalInsurance,
        "price" -> price,
        "check_in_date" -> proto.toLocalDate(offer.getDate).toString,
        "nights" -> offer.getNights,
        "tourist_group" -> Json.obj(
          "adults" -> ages.adults,
          "kids_ages" -> ages.kidsAgesWithoutInfants,
          "infants" -> ages.infants
        ),
        "meal" -> Json.obj(
          "code" -> offer.getPansion.toString,
          "name" -> offer.getRawPansion
        ),
        "ya_resort" -> hotel.geoId,
        "ya_country" -> geoBaseCountry.id,
        "ya_departure_city" -> request.hotelRequest.from,
        "ya_hotel" -> hotel.id,
        "ya_operator" -> operator.id,
        "room" -> Json.obj(
          "code" -> offer.getOriginalRoomCode,
          "name" -> offer.getRoomType
        )
      ) ++ JsObject(optFields)
    }
  }

  private def serializeFlight(flight: Flight) = {
    Json.obj(
      "to" -> serializeRoute(flight.getTo),
      "back" -> serializeRoute(flight.getBack),
      "additional_price" -> flight.getAdditionalPrice
    )
  }

  private def serializeRoute(route: Route) = {
    Json.obj(
      "route" -> route.getPointList.asScala.sliding(2).toVector.map(serializePoint),
      "direct" -> route.getIsDirect
    )
  }


  private def serializePoint(points: Seq[FlightPoint]): JsObject = {
    val Seq(from, to) = points
    val res = Json.obj(
      "source_code" -> from.getAirportCode,
      "source_airport_name" -> from.getAirport,
      "destination_code" -> to.getAirportCode,
      "destination_airport_name" -> to.getAirport,
      "departure" -> from.getDeparture,
      "arrival" -> to.getArrival
    )
    val optionalFields = mutable.Buffer.empty[(String, JsValue)]
    if (from.hasFlightNumber) optionalFields += "flight_no" -> JsString(from.getFlightNumber)
    if (from.hasCompany) optionalFields += "airline" -> JsString(from.getCompany)
    res ++ JsObject(optionalFields)
  }

  private def generateActualizedJson(request: OfferSearchRequest,
                                     actualizedOffer: ActualizedOffer,
                                     agency: OnlineTourAgency): Option[JsObject] = {
    generateOfferJson(request, actualizedOffer, agency).map { json ⇒
      if (actualizedOffer.hasActualizationInfo) {
        val ai = actualizedOffer.getActualizationInfo
        val optionalFields = mutable.Buffer.empty[(String, JsValue)]
        if (ai.hasVisaPrice) optionalFields += "visa" → JsNumber(ai.getVisaPrice)
        if (ai.hasInfantPrice) optionalFields += "infant_price" → JsNumber(ai.getInfantPrice)
        json ++ Json.obj(
          "fuel_charge" -> ai.getFuelCharge,
          "flights" -> ai.getFlightsList.asScala.map(serializeFlight)
        ) ++ JsObject(optionalFields)
      } else {
        json
      }
    }
  }

  private def getSignature(json: String, agency: OnlineTourAgency) = {
    Encryption.sha256hmac(agency.secretKey, json)
  }

  private def toIntJs(name: String, value: Option[String]): Seq[(String, JsValue)] = {
    value.flatMap(IntValue.apply).map(intValue ⇒ name → JsNumber(intValue)).toSeq
  }
}