package ru.yandex.tours.ota

import ru.yandex.tours.model.purchase.{PurchaseObject, PurchaseWithGet, PurchaseWithPost}
import ru.yandex.tours.util.Logging
import ru.yandex.vertis.billing.model.Properties

object AuctionService {
  val DEFAULT_CLICK_REVENUE = 3000l
}

/**
 * https://beta.wiki.yandex-team.ru/tourism/201415/product/tz/cpcauctionspecification/
 * 1. Разбиваем блок "Купить онлайн" на три части:
        1.1. Жёлтая кнопка
        1.2. @secondBlockSize белых кнопок
        1.3. Все остальные, скрытые за "Ещё"
   2. Всех партнёров сортируем по CPM
   3. Партнёра для жёлтой кнопки выбираем по максимальному CPM
   4. Для второго блока:
        4.1. выбираем двух следующих партнёров (из отсортированного по CPM списка)
        4.2. сортируем выбранных партнёров по CPC
   5. Партнёров для третьего блока выбираем по логике из п. 4
   6. В итоге получается список от 1 до 10 партнёров.
   7. Стоимость клика на кнопку партнёра  рассчитывается, как стоимость ставки партнёра, где n - позиция в полученном списке.
   8. Если среди онлайн-партнёров есть туроператор, для карточки тура которого рассчитывается список партнёров то:
      если туроператор не попадает на первое место, мы добавляем отельный блок "На сайте туроператора" и выводим кнопку
      туроператора в этот блок. (В остальных случаях, эта кнопка не показывается).


   Для CPA партнеров используется расчитанный CPC по итогам некоторого периода. Если туропертор находится не на первом месте,
   то стоимость клика равна его ставке.
 *
 */

class AuctionService(partnerStatistic: PartnerStatisticHolder) extends Logging {

  import AuctionService._

  def handle(purchaseObjects: Iterable[PurchaseObject], secondBlockSize: Int): Seq[PurchaseObject] = {
    if (purchaseObjects.isEmpty) {
      Seq.empty
    } else {
      val withCpm = purchaseObjects.map(o => (o, getClickRevenue(o) * partnerStatistic.getConversion(o.billingId)))
      val sorted = withCpm.toSeq.sortBy(-_._2).map(_._1)
      val first = sorted.head
      val (operators, tail) = sorted.tail.partition(_.isTourOperator)
      val second = tail.take(secondBlockSize).sortBy(getClickRevenue)(Ordering[Long].reverse)
      val third = tail.drop(secondBlockSize).sortBy(getClickRevenue)(Ordering[Long].reverse)
      val queue = Vector(first) ++ second ++ third
      if (log.isDebugEnabled) {
        log.debug("Queue:")
        dump(queue)
        log.debug("Operators:")
        dump(operators)
      }
      val withClickRevenue = queue.zipWithIndex.map { case (po, index) =>
        val clickRevenue = if (index < queue.size - 1) {
          getClickRevenue(queue(index + 1))
        } else {
          0
        }
        updateClickRevenue(po, clickRevenue)
      }
      val operatorsWithRevenue = operators.map(po => updateClickRevenue(po, getClickRevenue(po)))
      val result = Seq(withClickRevenue.head) ++ operatorsWithRevenue ++ withClickRevenue.tail
      if (log.isDebugEnabled) {
        log.debug("Result:")
        dump(result)
      }
      result
    }
  }

  private def dump(objects: Iterable[PurchaseObject]): Unit = {
    objects.foreach { obj =>
      val conversion = partnerStatistic.getConversion(obj.billingId)
      val cpc = getClickRevenue(obj)
      val finalRevenue = obj.billing.uiInfo.find(_._1 == Properties.BILLING_CLICK_REVENUE).map(_._2)
      log.debug(s"${obj.billingId} $cpc * $conversion = ${cpc * conversion} ($finalRevenue)")
    }
  }

  private def updateClickRevenue(po: PurchaseObject, clickRevenue: Long) = {
    val billing = po.billing.addUiInfo(Properties.BILLING_CLICK_REVENUE -> clickRevenue.toString)
    po match {
      case x: PurchaseWithGet => x.copy(billing = billing)
      case x: PurchaseWithPost => x.copy(billing = billing)
    }
  }

  private def getClickRevenue(po: PurchaseObject) = {
    po.billing.clickRevenue.orElse(partnerStatistic.getCalculatedCpc(po.billingId)).getOrElse(DEFAULT_CLICK_REVENUE)
  }
}
