package ru.yandex.tours

import akka.routing.RoundRobinPool
import ru.yandex.tours.app._
import ru.yandex.tours.calendar.LocalCalendarPushService
import ru.yandex.tours.calendar.storage.{FlightsStorage, PriceStorage}
import ru.yandex.tours.clickhouse.ClickHouseSupport
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.storage.direction._
import ru.yandex.tours.storage.minprice._
import ru.yandex.tours.storage.yt.entities.{DirectionPrice, Price}
import ru.yandex.tours.util.spray.{Replicator, RouteesContext}
import ru.yandex.tours.util.zoo.{BooleanSerializer, SharedValue}
import ru.yandex.travel.yt.daos.MirroredYtDao

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.03.16
 */
trait CalendarComponents {
  self: Application
    with AkkaSupport
    with SpraySupport
    with CommonDataHolders
    with ExtDataSupport
    with ClickHouseSupport
    with ZookeeperSupport
    with LocalCalendarServiceSupport
    with HttpSupport
    with YtSupport =>

  import akkaSystem.dispatcher

  private implicit val akka = akkaSystem

  lazy val routeesContext = new RouteesContext(departures, regionTree, None)(dispatcher)

  lazy val tourPriceStorage = new PriceStorage(clickHouseClient, hotelsIndex,
    SearchType.TOURS, regionTree, geoMappingHolder, searchSettings)

  lazy val roomPriceStorage = new PriceStorage(clickHouseClient, hotelsIndex,
    SearchType.ROOMS, regionTree, geoMappingHolder, searchSettings)

  lazy val flightsStorage = new FlightsStorage()

  lazy val enableWrites = new SharedValue[Boolean](zkClient, "/clickhouse/calendar_wr_en", false, BooleanSerializer)

  lazy val calendarPushService = new LocalCalendarPushService(
    tourPriceStorage, roomPriceStorage, flightsStorage,
    enableWrites)

//  lazy val toursMinPriceService = new ClickHouseMinPriceService(clickHouseClient, SearchType.TOURS, tourOperators)
//  lazy val roomsMinPriceService = new ClickHouseMinPriceService(clickHouseClient, SearchType.ROOMS, hotelProviders)

  lazy val toursMinPriceService = {
    new LocalMinPriceService(
      new MinPriceStorage(buildMinPriceDao()),
      buildDirectionPriceDao("direction_price")
    )
  }
  
  lazy val roomsMinPriceService = {
    new LocalMinPriceService(
      new MinPriceStorage(buildMinPriceDao("hotels_min_price")),
      buildDirectionPriceDao("hotels_direction_price")
    )
  }

  val apiProps = {
    CalendarApiHandler.props(
      routeesContext, calendarService, calendarPushService,
      toursMinPriceService, roomsMinPriceService,
      enableWrites, enableReads
    ).withRouter(RoundRobinPool(nrOfInstances = 16))
  }

  val apiHandler = akkaSystem.actorOf(apiProps, "api-handler")

  val replicatorProps = Replicator.props(
    apiHandler,
    self.config.getConfig("replicator"))

  onStart {
    log.info("Current clickhouse writing status: " + (if (enableWrites.get) "Enabled" else "Disabled"))
    enableWrites.onUpdate { enabled ⇒
      log.info("Updated clickhouse writing status: " + (if (enabled) "Enabled" else "Disabled"))
    }
    log.info("Current clickhouse reading status: " + (if (enableReads.get) "Enabled" else "Disabled"))
    enableReads.onUpdate { enabled ⇒
      log.info("Updated clickhouse reading status: " + (if (enabled) "Enabled" else "Disabled"))
    }
    startServer(replicatorProps, config.getInt("component.port"))
  }

  onStop {
    enableWrites.close()
    enableReads.close()
  }

  private def buildDirectionPriceDao(tableName: String): DirectionPriceDao = {
    lazy val ytDao = if (ytSettings.cacheDirections) {
      val inner = new YtDirectionPriceDao(createMirroredYtDao(
        classOf[DirectionPrice],
        ytSettings.primaryCluster,
        ytSettings.secondaryCluster,
        tableName,
        1.day.toMillis))
      new LRUCacheDirectionPriceDao(inner, 150000)
    }
    else {
      new YtDirectionPriceDao(
        createMirroredYtDao(
          classOf[DirectionPrice],
          ytSettings.primaryCluster,
          ytSettings.secondaryCluster,
          tableName,
          1.day.toMillis)
      )
    }
    ytDao
  }

  private def buildMinPriceDao(table_name: String = null): MinPriceDao = {
    val yt_name = if (table_name == null) "min_price" else table_name
    lazy val ytDao = {
      new YtMinPriceDao(createMirroredYtDao(
        classOf[Price],
        ytSettings.primaryCluster,
        ytSettings.secondaryCluster,
        yt_name,
        1.day.toMillis))
    }
    ytDao
  }
}
