package ru.yandex.tours.calendar

import org.joda.time.format.DateTimeFormat
import org.joda.time.{Days, LocalDate}
import ru.yandex.tours.clickhouse.ClickHouseClient
import ru.yandex.tours.model.Prices.{DirectionBestPrice, PriceGraph}
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{HotelSearchRequest, SearchType}
import ru.yandex.tours.model.util.DateInterval
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.services.MinPriceService
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.parsing.{IntValue, LocalDateValue}

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 08.04.16
 */
class ClickHouseMinPriceService(clickHouseClient: ClickHouseClient, searchType: SearchType,
                                sources: SearchSources[_ <: Source])
                               (implicit ec: ExecutionContext) extends MinPriceService {
  private case class PriceEntity(when: LocalDate, operatorId: Int, price: Int)

  private val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")

  private val prefix = searchType match {
    case SearchType.TOURS => "tour"
    case SearchType.ROOMS => "room"
  }

  private def dateIn(column: String, dateInterval: DateInterval) = {
    import dateInterval._
    s"($column >= \'${start.toString(dateFormat)}\') AND ($column <= \'${end.toString(dateFormat)}\')"
  }

  private def q(from: Int, to: Int, nights: Int, ages: Seq[Int], whenInterval: DateInterval) = {
    val where =
      s"""from = $from
          |AND to = $to
          |AND ages = '${ages.mkString(",")}'
          |AND ${dateIn("when", whenInterval)}
          |AND nights = $nights
          |AND when >= today()
          |AND timestamp >= now() - 86400
       """.stripMargin

    val prices =
      s"""SELECT
          |  when, operator_id, hotel_id,
          |  any(price_timestamp) AS price_timestamp,
          |  minArray(hotel_prices) min_price
          |FROM (
          |  SELECT
          |    when, operator_id, hotel_id,
          |    max(timestamp) AS price_timestamp,
          |    argMax(`prices.price`, timestamp) AS hotel_prices
          |  FROM ${clickHouseClient.database}.${prefix}_prices
          |  WHERE $where
          |  GROUP BY when, operator_id, hotel_id
          |)
          |GROUP BY when, operator_id, hotel_id
       """.stripMargin

    val empty =
      s"""SELECT
          |  when, operator_id,
          |  max(timestamp) AS empty_timestamp
          |FROM ${clickHouseClient.database}.${prefix}_empty_results
          |WHERE $where
          |GROUP BY when, operator_id
       """.stripMargin

    s"""SELECT
        |  when, operator_id,
        |  minIf(min_price, min_price > 0) = 0 ? -2 : minIf(min_price, min_price > 0)
        |FROM (
        |  SELECT
        |    when,
        |    operator_id,
        |    hotel_id,
        |    (price_timestamp < empty_timestamp) ? -2 : min_price AS min_price
        |  FROM (
        |    $prices
        |  )
        |  ANY LEFT JOIN (
        |    $empty
        |  ) USING when, operator_id
        |)
        |GROUP BY when, operator_id
     """.stripMargin
  }

  private def getPriceEntities(from: Int, to: Int, nights: Int, ages: Seq[Int],
                               graphStart: LocalDate, graphEnd: LocalDate): Future[Array[PriceEntity]] = {
    val query = q(from, to, nights, ages, DateInterval(graphStart, graphEnd))

    clickHouseClient.query(query).map { lines =>
      lines.map {
        case Seq(LocalDateValue(w), IntValue(operatorId), IntValue(price)) =>
          PriceEntity(w, operatorId, price)
      }.toArray
    }
  }

  override def getMinPriceGraph(request: HotelSearchRequest, graphLength: Int): Future[PriceGraph] = {
    val start = request.when.minusDays(graphLength - 1)
    val end = request.when.plusDays(graphLength - 1)

    for {
      prices <- getPriceEntities(request.from, request.to, request.nights, request.agesSerializable, start, end)
    } yield {
      val today = LocalDate.now
      val startDate = getStartDate(prices, today, request.when, graphLength)
      val pricesForGraph = prices.filter(isInSameGraph(startDate, graphLength))
      buildGraph(request.from, request.to, request.when, request.nights, request.agesSerializable,
        startDate, pricesForGraph, graphLength)
    }
  }

  override def getMinPriceGraph(from: Int, to: Int,
                                when: LocalDate, nights: Int, ages: Seq[Int],
                                graphStart: LocalDate, graphEnd: LocalDate): Future[PriceGraph] = {
    for {
      prices <- getPriceEntities(from, to, nights, ages, graphStart, graphEnd)
    } yield {
      buildGraph(from, to, when, nights, ages, graphStart, prices, Days.daysBetween(graphStart, graphEnd).getDays + 1)
    }
  }

  override def getDirectionMinPrice(from: Int, to: Int): Future[Option[DirectionBestPrice]] = {
    val q =
      s"""SELECT
         |  *
         |FROM
         |
       """.stripMargin

    Future.successful(None)
  }

  /*
    Here we want to choose date, which will be the start of the graph
    Also, there should be some gap between "start date" / "end date" and "requested date",
    because we don't want "requested date" to be at to the end of the graph.
    But if "today" is near to "requested date", start date should be "today"
    P.S
      "start date" - start of the graph
      "end date" - end of the graph
      "requested date" - "when" date from request
   */
  private def getStartDate(actualPrices: Iterable[PriceEntity], today: LocalDate, when: LocalDate, graphLength: Int) = {
    val gapLength = graphLength / 4
    if (Days.daysBetween(today, when).getDays <= graphLength - gapLength) {
      today
    } else {
      val maxPriceCountShift = (-(graphLength - gapLength) to -gapLength).maxBy(dayShift => {
        val startDay = when.plusDays(dayShift)
        actualPrices.count(isInSameGraph(startDay, graphLength))
      })
      when.plusDays(maxPriceCountShift)
    }
  }

  /*
  Checks whether @otherPrice would be in the graph with starting at @start with length @graphLength
 */
  private def isInSameGraph(start: LocalDate, graphLength: Int)(otherPrice: PriceEntity) = {
    val when = otherPrice.when
    (when >= start) && Days.daysBetween(start, when).getDays.abs < graphLength
  }


  private def mergeMinPrices(prices: Seq[PriceEntity], operators: Set[Int]): Int = {
    if (prices.isEmpty) MinPriceService.UNKNOWN_PRICE
    else {
      val withPrices = prices.filter(_.price > 0)
      if (withPrices.isEmpty) {
        if (prices.map(_.operatorId).toSet == operators) MinPriceService.NO_TOURS
        else MinPriceService.UNKNOWN_PRICE
      } else withPrices.map(_.price).min
    }
  }

  private def buildOperatorPrice(prices: Seq[PriceEntity], operators: Set[Int]) = {
    val builder = PriceGraph.PriceEntity.newBuilder()
    for (price <- prices) yield {
      builder.addOperatorPriceBuilder()
        .setOperatorId(price.operatorId)
        .setPrice(price.price)
    }
    builder.setMinPrice(mergeMinPrices(prices, operators)).build()
  }

  private def buildGraph(from: Int, to: Int, when: LocalDate, nights: Int, ages: Seq[Int],
                         startDate: LocalDate,
                         prices: Iterable[PriceEntity],
                         graphLength: Int) = {
    require(prices.forall(_.when >= startDate), s"Expected startDate be <= to all entities in graph")
    val dayToPrice = prices.map(t => Days.daysBetween(startDate, t.when).getDays.abs -> t).toMultiMap
      .withDefaultValue(Seq.empty)

    val operators = prices.map(_.operatorId).toSet
    val priceArray = (0 until graphLength).map(dayToPrice).map(mergeMinPrices(_, operators)).map(Int.box)
    val operatorPriceArray = (0 until graphLength).map(dayToPrice).map(buildOperatorPrice(_, operators))
    PriceGraph.newBuilder()
      .addAllOBSOLETEPrice(asJavaIterable(priceArray))
      .addAllPriceEntity(asJavaIterable(operatorPriceArray))
      .setIndex(Days.daysBetween(startDate, when).getDays.abs)
      .setWhen(when.toMillis)
      .addAllAge(ages.map(Int.box))
      .setNights(nights)
      .setFrom(from)
      .setTo(to)
      .build()
  }
}
