package ru.yandex.tours.calendar.storage

import org.joda.time.format.DateTimeFormat
import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.calendar.storage.PriceStorage.{HotelPrice, NoPrices, Price}
import ru.yandex.tours.clickhouse.ClickHouseClient
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchResults.{HotelSearchResult, OfferSearchResult}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest, SearchType}
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.tours.util.parsing.Tabbed

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.03.16
 */
class PriceStorage(clickHouseClient: ClickHouseClient,
                   hotelsIndex: HotelsIndex,
                   searchType: SearchType,
                   tree: Tree,
                   geoMapping: GeoMappingHolder,
                   searchSettingsHolder: SearchSettingsHolder)(implicit ec: ExecutionContext) {

  private val prefix = searchType match {
    case SearchType.TOURS => "tour"
    case SearchType.ROOMS => "room"
  }

  val insertPricesQuery =
    s"""insert into ${clickHouseClient.database}.${prefix}_prices(
      |  timestamp,
      |  from, to,
      |  when, nights, ages,
      |  hotel_id,
      |  operator_id,
      |  `prices.pansion`, `prices.price`
      |) FORMAT TabSeparated""".stripMargin

  val insertEmptyResultsQuery =
    s"""insert into ${clickHouseClient.database}.${prefix}_empty_results(
      |  timestamp,
      |  from, to,
      |  when, nights, ages,
      |  operator_id
      |) FORMAT TabSeparated""".stripMargin

  def normalize(req: HotelSearchRequest): HotelSearchRequest = {
    searchType match {
      case SearchType.TOURS => req
      case SearchType.ROOMS => req.copy(from = 0, flexWhen = false, flexNights = false)
    }
  }

  def saveSnippets(req: HotelSearchRequest, res: HotelSearchResult): Future[Unit] = {
    val (prices, noPrices) = partition(normalize(req), res)
    (savePrices(prices) zip saveNoPrices(noPrices)).toUnit
  }

  def saveOffers(req: OfferSearchRequest, res: OfferSearchResult): Future[Unit] = {
    val (prices, noPrices) = partition(req, res)
    (savePrices(prices) zip saveNoPrices(noPrices)).toUnit
  }

  def savePrices(prices: Seq[HotelPrice]): Future[Unit] = {
    if (prices.isEmpty) return Future.successful(())
    clickHouseClient.update(insertPricesQuery, prices.map(_.toRowString))
  }

  def saveNoPrices(emptyPrices: Seq[NoPrices]): Future[Unit] = {
    if (emptyPrices.isEmpty) return Future.successful(())
    clickHouseClient.update(insertEmptyResultsQuery, emptyPrices.map(_.toRowString))
  }

  private def regions(hotel: Hotel): Set[Int] = {
    tree.findParents(hotel.geoId).map(_.id)
      .filter(geoMapping.isKnownDestination)
      .filter(searchSettingsHolder.getRegionSearchSettings(_).isAllowed(searchType))
  }

  def emptyHotelPrices(req: HotelSearchRequest,
                       hotel: Hotel,
                       operatorIds: Set[Int],
                       dateTime: DateTime): Seq[HotelPrice] = {
    if (operatorIds.isEmpty) return Seq.empty

    for {
      operatorId <- operatorIds.toSeq
      when <- req.dateRange
      nights <- req.nightsRange
      to <- regions(hotel)
    } yield {
      HotelPrice(
        dateTime,
        req.from, to,
        when, nights, req.agesSerializable,
        hotel.id,
        operatorId,
        Seq.empty
      )
    }
  }

  def hotelPrices(req: HotelSearchRequest,
                  hotel: Hotel,
                  offers: Seq[Offer],
                  dateTime: DateTime): Seq[HotelPrice] = {
    for {
      ((date, nights, operatorId), offers) <- offers.groupBy(o => (o.getDate, o.getNights, o.getSource.getOperatorId))
      to <- regions(hotel)
    } yield {
      HotelPrice(
        dateTime,
        req.from, to,
        date.toLocalDate,
        nights,
        req.agesSerializable,
        hotel.id,
        operatorId,
        offers.map(o => Price(o.getPansion, o.getPrice))
      )
    }
  }.toSeq

  def regionPrices(req: HotelSearchRequest,
                   snippets: Seq[HotelSnippet],
                   dateTime: DateTime): Seq[HotelPrice] = {

    val hotelsMap = hotelsIndex.getHotelsById(snippets.map(_.getHotelId).toSet)
    for {
      (operatorId, operatorSnippets) <- snippets.groupBy(_.getSourceList.asScala.head.getOperatorId)
      ((hotelId, date, nights), snippets) <- operatorSnippets.groupBy(s => (s.getHotelId, s.getDateMin, s.getNightsMin))
      hotel <- hotelsMap.get(hotelId).toSeq
      to <- regions(hotel) + req.to
    } yield {
      val prices = for {
        snippet <- snippets
        pricedPansion <- snippet.getPansionsList.asScala
      } yield Price(pricedPansion.getPansion, pricedPansion.getPrice)

      HotelPrice(
        dateTime,
        req.from, to,
        date.toLocalDate,
        nights,
        req.agesSerializable,
        hotel.id,
        operatorId,
        prices
      )
    }
  }.toSeq

  def noPrices(req: HotelSearchRequest,
               operatorIds: Set[Int],
               dateTime: DateTime): Seq[NoPrices] = {
    if (operatorIds.isEmpty) return Seq.empty
    for {
      when <- req.dateRange
      nights <- req.nightsRange
      operatorId <- operatorIds
    } yield {
      NoPrices(
        dateTime,
        req.from, req.to,
        when, nights, req.agesSerializable,
        operatorId
      )
    }
  }

  def partition(req: HotelSearchRequest, res: HotelSearchResult): (Seq[HotelPrice], Seq[NoPrices]) = {
    val dateTime = new DateTime(res.getUpdated)
    val snippets = res.getHotelSnippetList.asScala
    val completeOperatorSet = SimpleBitSet.from(res.getProgress.getOperatorCompleteSet).toSet
    val foundOperators = snippets.flatMap(_.getSourceList.asScala).map(_.getOperatorId).toSet
    val emptyOperators = completeOperatorSet -- foundOperators

    regionPrices(req, snippets, dateTime) -> noPrices(req, emptyOperators, dateTime)
  }

  def partition(req: OfferSearchRequest, res: OfferSearchResult): (Seq[HotelPrice], Seq[NoPrices]) = {
    val hr = normalize(req.hotelRequest)
    val dateTime = new DateTime(res.getUpdated)

    hotelsIndex.getHotelById(req.hotelId) match {
      case Some(hotel) =>
        if (res.getOfferCount == 0) {
          emptyHotelPrices(hr, hotel,
            SimpleBitSet.from(res.getProgress.getOperatorCompleteSet).toSet, dateTime) → Seq.empty
        } else {
          val completeOperatorSet = SimpleBitSet.from(res.getProgress.getOperatorCompleteSet).toSet
          val offers = res.getOfferList.asScala
          val foundOperators = offers.map(offer => offer.getSource.getOperatorId).toSet

          val emptyOperators = completeOperatorSet -- foundOperators
          val emptyPrices = emptyHotelPrices(hr, hotel, emptyOperators, dateTime)
          val nonEmptyPrices = hotelPrices(hr, hotel, offers, dateTime)
          (emptyPrices ++ nonEmptyPrices) -> Seq.empty
        }
      case None =>
        Seq.empty -> Seq.empty
    }
  }
}

object PriceStorage {

  private val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")
  private val dateTimeFormat = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss")


  case class Price(pansion: Pansion, price: Int)

  case class HotelPrice(dateTime: DateTime,
                        from: Int,
                        to: Int,
                        when: LocalDate,
                        nights: Int,
                        ages: Seq[Int],
                        hotelId: Int,
                        operatorId: Int,
                        prices: Seq[Price]) {
    def toRowString: String = Tabbed(
      dateTime.toString(dateTimeFormat),
      from, to,
      when.toString(dateFormat), nights, ages.sorted.mkString(","),
      hotelId,
      operatorId,
      prices.map(_.pansion.getNumber).mkString("[", ",", "]"),
      prices.map(_.price).mkString("[", ",", "]")
    )
  }

  case class NoPrices(dateTime: DateTime,
                      from: Int,
                      to: Int,
                      when: LocalDate,
                      nights: Int,
                      ages: Seq[Int],
                      operatorId: Int) {
    def toRowString: String = Tabbed(
      dateTime.toString(dateTimeFormat),
      from, to,
      when.toString(dateFormat), nights, ages.sorted.mkString(","),
      operatorId
    )
  }

}