package ru.yandex.tours.agencies

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.metro.{MetroHolder, MetroUtil}
import ru.yandex.tours.serialize.json.CommonJsonSerialization._
import ru.yandex.tours.util.spray._
import spray.http.StatusCodes
import spray.routing.Route

class AgenciesHandler(agencies: Agencies, agencyIndex: AgenciesIndex, metroHolder: MetroHolder)
  extends HttpHandler with CommonModelDirectives {

  private val isOpened = parameter('is_opened.as[Boolean] ? false)
  private val size = parameter('size.as[Int])
  private val preferTo = parameter('prefer_to.as[Int].?)
  private val recommendedSize = parameter('rec_size.as[Int] ? 5)
  private val maxSize = parameter('max_size.as[Int] ? 200)

  private val mapInfo: Route = {
    (path("map_info") & mapRectangle & lang & isOpened & maxSize) {
      (mapRectangle, lang, isOpened, maxSize) ⇒
        var agencies = agencyIndex.inRectangle(mapRectangle)
        if (isOpened) {
          agencies = agencies.filter(_.agency.currentWorkTimeStatus.exists(_.isOpen))
        }
        agencies = agencies.take(maxSize)
        val ans = new JSONObject()
          .put("agencies", agenciesToJson(agencies, metroHolder, lang))
          .put("map", toJson(mapRectangle))
        completeJsonOk(ans)
    }
  }

  private val agencyById: Route = {
    (path(LongNumber) & lang) { (agencyId, lang) ⇒
      agencies.get(agencyId) match {
        case Some(agency) ⇒ completeJsonOk(agencyToJson(agency, metroHolder, lang, None))
        case None ⇒ completeJsonError(StatusCodes.NotFound, "Unknown agency id")
      }
    }
  }

  private val getAgencies = {
    (pathEndOrSingleSlash & lang & geoId & size & preferTo & recommendedSize & optMetro(metroHolder)) {
      (lang, geoId, size, preferredOperator, recommendedSize, optMetro) ⇒
        val agencies = optMetro match {
          case Some(metro) ⇒
            agencyIndex.getSampleAndCountNearMetro(geoId, MetroUtil.compositeId(metro),
              size, recommendedSize, preferredOperator)
          case None ⇒
            agencyIndex.getSampleAndCountInRegion(geoId, size, recommendedSize, preferredOperator)
        }
        val ans = toJson(agencies, metroHolder, lang, optMetro.map(_.getColor))
        optMetro.foreach(m ⇒ ans.put("metro", MetroUtil.compositeId(m)))
        completeJsonOk(ans)
    }
  }
  private val agenciesByIds = {
    (pathEndOrSingleSlash & lang & longArray("id")) { (lang, ids) ⇒
      val ar = new JSONArray()
      ids.flatMap(agencies.get).foreach {
        agency ⇒ ar.put(agencyToJson(agency, metroHolder, lang, metroColor = None))
      }
      completeJsonOk(ar)
    }
  }

  override val route = mapInfo ~ agencyById ~ getAgencies ~ agenciesByIds
}
