package ru.yandex.tours.agencies

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.agencies.AgenciesIndex.AgencyWithBilling
import ru.yandex.tours.billing.BillingIndex
import ru.yandex.tours.billing.BillingOffers.IdedOfferBilling
import ru.yandex.tours.extdata.{CompositeDataDef, DataTypes}
import ru.yandex.tours.geo.GeoIndex
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.{Agency, MapRectangle}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging
import shapeless._

import scala.util.{Random, Try}

class AgenciesIndex(val agencies: Iterable[AgencyWithBilling],
                    tree: Tree,
                    partneringIndex: AgencyPartneringIndex,
                    customerToOperatorIndex: CustomerToOperatorIndex) extends GeoIndex(agencies) {

  import AgenciesIndex._

  private val (paid, regular) = agencies.partition(_.isPaid)

  private val id2agency = agencies.map(t => t.agency.id → t).toMap
  private val geoIdToPaid = grouped(paid, pathToCountry)
  private val geoIdToRegular = grouped(regular, pathToCountry, TOP_SIZE)
  private val metroIdToPaid = grouped(paid, _.agency.metroIds)
  private val metroIdToRegular = grouped(regular, _.agency.metroIds)

  def getById(id: Long): Option[AgencyWithBilling] = id2agency.get(id)

  def getSampleAndCountInRegion(geoId: Int,
                                size: Int = SAMPLE_SIZE,
                                recommendedSize: Int = SAMPLE_SIZE,
                                preferredOperator: Option[Int]): AgenciesInRegion = {
    tree.region(geoId).map { region =>
      val paid = geoIdToPaid(region.id)
      val regular = geoIdToRegular(region.id)
      val (recommended, rest) = getSample(paid, regular, size, recommendedSize, preferredOperator)
      AgenciesInRegion(Some(region), recommended, rest, totalSize(region.id), totalSize(region.id))
    }.getOrElse {
      emptyAgenciesInRegion
    }
  }

  def getSampleAndCountNearMetro(geoId: Int,
                                 metroId: String,
                                 size: Int = SAMPLE_SIZE,
                                 recommendedSize: Int = SAMPLE_SIZE,
                                 preferredOperator: Option[Int]): AgenciesInRegion = {
    tree.region(geoId) match {
      case Some(region) =>
        val paid = metroIdToPaid(metroId)
        val regular = metroIdToRegular(metroId)
        val (recommended, rest) = getSample(paid, regular, size, recommendedSize, preferredOperator)
        AgenciesInRegion(Some(region), recommended, rest, paid.size + regular.size, totalSize(geoId))
      case None => emptyAgenciesInRegion
    }
  }

  override def inRectangle(mapInfo: MapRectangle): Seq[AgencyWithBilling] = {
    val agencies = super.inRectangle(mapInfo)
    val (paid, regular) = agencies.partition(_.isPaid)
    Random.shuffle(paid) ++ regular
  }

  private val emptyAgenciesInRegion = AgenciesInRegion(None, Iterable.empty, Iterable.empty, 0, 0)

  private def totalSize(geoId: Int): Int = geoIdToPaid(geoId).size + geoIdToRegular(geoId).size

  private def getSample(paid: Seq[AgencyWithBilling],
                        regular: Seq[AgencyWithBilling],
                        size: Int,
                        recommendedSize: Int,
                        preferredOperator: Option[Int]) = {
    val filteredPaid = filteredNotPartnered(paid, preferredOperator)
    val filteredRegular = filteredNotPartnered(regular, preferredOperator)
    val (recommended, restPaid) = preferredOperator.flatMap(customerToOperatorIndex.getClient) match {
      case Some(customer) ⇒
        filteredPaid.partition(isClientIdEqualTo(customer))
      case None =>
        (Seq.empty[AgencyWithBilling], filteredPaid)
    }
    val shuffled = Random.shuffle(restPaid)
    val rest = if (shuffled.size >= size) {
      shuffled.take(size)
    } else {
      val nonPaid = filteredRegular.sortBy(-_.agency.shows).take(TOP_SIZE)
      shuffled ++ Random.shuffle(nonPaid).take(size - shuffled.size)
    }
    (Random.shuffle(recommended).take(recommendedSize), rest)
  }

  private def isClientIdEqualTo(clientId: Long)(agency: AgencyWithBilling): Boolean = {
    Try(agency.billing.offer.getKnownCampaign.getCampaign.getOwner.getId.getClientId == clientId).getOrElse(false)
  }

  private def pathToCountry(agencyWithBilling: AgencyWithBilling): Seq[Int] = {
    val geoId = agencyWithBilling.agency.geoId
    val response = tree.pathToRoot(geoId).takeWhile(!_.isCountry).map(_.id)
    if (response.isEmpty) {
      Seq(geoId)
    } else {
      response
    }
  }

  private def filteredNotPartnered(agencies: Seq[AgencyWithBilling], preferredOperator: Option[Int]) = {
    preferredOperator match {
      case Some(to) => agencies.filterNot(agency => partneringIndex.getNotPartnered(agency.agency.id).contains(to))
      case None => agencies
    }
  }

  private def grouped[T](agencies: Iterable[AgencyWithBilling],
                         keys: AgencyWithBilling => Iterable[T],
                         size: Int = Integer.MAX_VALUE): Map[T, Seq[AgencyWithBilling]] = {
    (for {
      agency <- agencies
      key <- keys(agency)
    } yield key -> agency).toMultiMap.map {
      case (geoId, x) => geoId -> x.sortBy(-_.agency.shows).take(size)
    }.withDefaultValue(Seq.empty)
  }
}

object AgenciesIndex extends CompositeDataDef[AgenciesIndex, Agencies :: Tree :: BillingIndex ::
  AgencyPartneringIndex :: CustomerToOperatorIndex :: HNil] with Logging {

  case class AgenciesInRegion(region: Option[Region],
                              recommended: Iterable[AgencyWithBilling],
                              sample: Iterable[AgencyWithBilling],
                              count: Int,
                              totalCount: Int)

  case class AgencyWithBilling(agency: Agency, billing: IdedOfferBilling) extends MapObject {
    override def longitude: Double = agency.longitude

    override def latitude: Double = agency.latitude

    def isPaid: Boolean = billing.offer.getKnownCampaign.getCampaign.getOrder.getProductKey match {
      case "offline-biz" => true
      case "verified-offline-biz" => true
      case _ => false
    }
  }

  private val SAMPLE_SIZE = 4
  private val TOP_SIZE = 30

  override def dependsOn: Set[DataType] = Set(DataTypes.offlineAgencyBilling)

  override def from(dependencies: Agencies :: Tree :: BillingIndex ::
    AgencyPartneringIndex :: CustomerToOperatorIndex :: HNil): AgenciesIndex = {
    val agenciesIndex :: tree :: billingIndex :: partneringIndex :: customer2operator :: HNil = dependencies
    val agencies = for {
      (id, billing) ← billingIndex.id2offer
      agency <- agenciesIndex.get(id)
    } yield AgencyWithBilling(agency, billing)
    new AgenciesIndex(agencies, tree, partneringIndex, customer2operator)
  }
}
