package ru.yandex.tours.app

import java.io.File
import java.lang.Thread.UncaughtExceptionHandler

import com.typesafe.config.Config
import ru.yandex.tours.util.Logging
import sun.misc.Signal

import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 21.01.15
 */
trait Application extends Logging {

  def config: Config

  def environmentType: String

  def isEnvironmentStable: Boolean = environmentType == "production"
  def isEnvironmentTesting: Boolean = environmentType == "testing"
  def isEnvironmentDevelopment: Boolean = environmentType == "development"
  def isEnvironmentLocal: Boolean = environmentType == "local"
  def isEnvironmentPrestable: Boolean = environmentType == "prestable"

  def serviceName: String
  def componentName: String

  def hostName: String
  def dataCenter: String

  def dataFolder: File
  def etcDir: File

  def start(interactive: Boolean): Unit = {
    try {
      log.info(s"$componentName is starting...")
      startHooks.foreach(_.apply())
      log.info(s"$componentName started")
      if (interactive) {
        this.interactive()
      }
    } catch {
      case t: Throwable =>
        halt(t)
    }
  }

  private def interactive() {
    println(27.toChar + "[31mPress Ctrl+D to exit" + 27.toChar + "[0m")
    while (true) {
      val code = System.in.read()
      if (code == -1) {
        stop()
        sys.exit(0)
      }
    }
  }

  def stop(): Unit = {
    log.info(s"$componentName is stopping...")
    stopHooks.foreach(_.apply())
    log.info(s"$componentName stopped")
  }

  private def halt(cause: Throwable, thread: Thread = Thread.currentThread()): Unit = {
    log.error(s"Fatal error in thread $thread", cause)
    System.err.println(s"Fatal error in thread $thread: ${cause.getMessage}")
    cause.printStackTrace(System.err)
    System.err.flush()
    sys.exit(1)
  }

  private var startHooks: Vector[() => Unit] = Vector.empty
  private var stopHooks: Vector[() => Unit] = Vector.empty
  protected def onStart(action: => Unit): Unit = {
    this.synchronized {
      startHooks :+= (() => action)
    }
  }
  protected def onStop(action: => Unit): Unit = {
    this.synchronized {
      stopHooks :+= (() => action)
    }
  }

  def main(args: Array[String]): Unit = {
    Thread.setDefaultUncaughtExceptionHandler(
      new UncaughtExceptionHandler {
        def uncaughtException(t: Thread, e: Throwable) {
          log.error(s"uncaught failure in thread $t", e)
          if (!NonFatal(e)) {
            halt(e, t)
          }
        }
      }
    )
    sys.addShutdownHook {
      stop()
    }

    Signal.handle(new Signal("HUP"), Logging)

    def isInteractive = config.hasPath(Application.interactiveName) && config.getBoolean(Application.interactiveName)
    start(isInteractive)
  }
}

object Application {
  private val interactiveName = "service.interactive"
}
