package ru.yandex.tours.app

import java.io.File

import ru.yandex.tours.avia.Airports
import ru.yandex.tours.currency.CurrencyRates
import ru.yandex.tours.direction._
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.geo.{Departures, Iso2Country, LtActiveCountries}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.base.region.RegionBoundaries
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels._
import ru.yandex.tours.operators.{HotelProviders, TourOperators}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.resorts.{Brandings, SkiResorts}
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.tanker.Translations
import shapeless._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.04.15
 */
trait CommonDataHolders { app: Application with ExtDataSupport with AkkaSupport =>

  def lazyLoadResources: Boolean = false

  lazy val hotelRatings = HotelRatings.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val regionBoundaries = RegionBoundaries.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val regionTree = region.Tree.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val hotelsIndex: HotelsIndex = {
    val conf = config.getConfig("tours.hotels.index")
    val hotelsDir = new File(conf.getString("dir"))
    val baseIndex = ShardedYoctoHotelsIndex.fromExtData(hotelsDir, hotelRatings, extDataService, extDataUpdateChecker, lazyLoadResources)
    new FilteringHotelsIndex(baseIndex, regionTree)
  }
  lazy val hotelsVideo = HotelsVideo.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val hotelsPanoramas = HotelsPanoramas.from(hotelsIndex :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val iso2country = Iso2Country.from(regionTree :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val geoMappingHolder = GeoMappingHolder.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val ltActiveCountries = LtActiveCountries
    .from(geoMappingHolder :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val searchSettings = SearchSettingsHolder.
    from(regionTree :: ltActiveCountries :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val brandings = Brandings.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val currencyRates = CurrencyRates.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val oldLtGeoMappingHolder = GeoMappingHolder.oldLtFrom(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val skiResorts = SkiResorts.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val directions = Directions.from(regionTree :: geoMappingHolder :: skiResorts :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val directionsSimilarity = DirectionsSimilarity.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val directionsStats = DirectionsStats.from(
    countryPriorities ::
      resortPriorities ::
      hotelsIndex :: regionTree :: geoMappingHolder :: directionsSimilarity :: HNil,
    extDataUpdateChecker, lazyLoadResources
  )
  lazy val hotDirections = HotDirections.from(extDataService, extDataUpdateChecker)

  lazy val departures = Departures.from(
    regionTree :: geoMappingHolder :: HNil,
    extDataService, extDataUpdateChecker, lazyLoadResources
  )

  lazy val translations = Translations.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val tourOperators = TourOperators.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val hotelProviders = HotelProviders.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val pansionUnifier = PansionUnifier.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val airports = Airports.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val hotelPriorities = HotelPriorities.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val countryPriorities = CountryPriorities.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val resortPriorities = ResortPriorities.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val frontHost = app.config.getString("tours.front.host")
  lazy val urlBuilder = new UrlBuilder(frontHost)

  lazy val hotelsSimilarity = HotelsSimilarity.from(
    hotelsIndex :: HNil,
    extDataService, extDataUpdateChecker, lazyLoadResources
  )

}
