package ru.yandex.tours.app

import java.io.{FileNotFoundException, File, FileInputStream}
import java.util.Properties

import com.typesafe.config.{Config, ConfigFactory}
import org.apache.commons.io.FileUtils

import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 15.01.15
 */
trait DefaultEnvironment { this: Application =>
  lazy val environmentType: String = Try(
    FileUtils.readFileToString(new File("/etc/yandex/environment.type"))
  ).getOrElse("local").trim

  lazy val serviceName = config.getString("service.name")

  lazy val componentName = config.getString("component.name")

  lazy val dataFolder = new File(config.getString("tours.data-folder"))

  lazy val etcDir = new File(s"/etc/yandex/$serviceName/$componentName")

  private def machineProperties = {
    val props = new Properties()
    try {
      props.load(new FileInputStream("/etc/yandex/machine.properties"))
    } catch {
      case ex: FileNotFoundException => log.warn("Can not load machine properties, using empty one")
    }
    props
  }

  lazy val hostName = machineProperties.getProperty("fqdn", System.getProperty("host.name", "undefined.host"))
  lazy val dataCenter = machineProperties.getProperty("root_datacenter", System.getProperty("dc.name", "undefined"))

  lazy val config: Config = {
    val resources = List(
      "application.local.conf",
      "application.docker.conf",
      "application.private.conf",
      "application.development.conf",
      "application.testing.conf",
      "application.conf"
    )
    val loadResources = environmentType match {
      case "production" => resources.drop(5)
      case "prestable" => resources.drop(5)
      case "testing" => resources.drop(4)
      case "development" => resources.drop(2)
      case "docker" => resources.drop(1)
      case _ => resources
    }

    val config = loadResources.foldLeft(ConfigFactory.empty()) {
      case (c, resource) =>
        log.info(s"Load properties from <$resource>")
        c withFallback Try(ConfigFactory.parseResources(resource)).getOrElse(ConfigFactory.empty())
    }
    ConfigFactory.load(config)
  }
}