package ru.yandex.tours.app

import com.codahale.metrics.jvm.{FileDescriptorRatioGauge, GarbageCollectorMetricSet, MemoryUsageGaugeSet, ThreadStatesGaugeSet}
import com.codahale.metrics._
import org.slf4j.LoggerFactory
import ru.yandex.common.monitoring.Metrics
import ru.yandex.common.monitoring.graphite.GraphiteReporterEnabler

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 24.02.15
 */
trait MetricsSupport { this: Application =>
  def metricsRegistry: MetricRegistry

  def registerAll(prefix: String, metrics: MetricSet): Unit = {
    import scala.collection.JavaConverters._
    for ((name, metric) <- metrics.getMetrics.asScala) {
      metricsRegistry.register(MetricRegistry.name(prefix, name), metric)
    }
  }
}

trait DefaultMetricsSupport extends MetricsSupport { this: Application =>
  override def metricsRegistry: MetricRegistry = Metrics.defaultRegistry()

  onStart {
    registerAll("jvm.memory", new MemoryUsageGaugeSet())
    registerAll("jvm.gc", new GarbageCollectorMetricSet())
    metricsRegistry.register("jvm.file", new FileDescriptorRatioGauge())
  }

  onStop {
    metricsRegistry.removeMatching(new MetricFilter {
      override def matches(name: String, metric: Metric): Boolean = true
    })
  }
}

trait GraphiteReporting { app: Application with MetricsSupport =>

  onStart {
    val logger = LoggerFactory.getLogger(classOf[GraphiteReporting])
    if (app.isEnvironmentTesting || app.isEnvironmentStable || app.isEnvironmentPrestable) {
      logger.info(s"Starting graphite reporting as $componentName in $hostName")
      GraphiteReporterEnabler.enable(metricsRegistry, componentName, hostName)
    } else {
      logger.info(s"Do not started graphite reporting in ${app.environmentType} env.")
    }
  }
}

trait JmxReporting { app: Application with MetricsSupport =>
  private val reporter = JmxReporter.forRegistry(metricsRegistry).build()

  onStart {
    reporter.start()
  }

  onStop {
    reporter.stop()
  }
}