package ru.yandex.tours.app

import org.apache.curator.framework.recipes.AfterConnectionEstablished
import ru.yandex.tours.util.zoo
import ru.yandex.vertis.curator.recipes.map.{LoggingZooKeeperMap, ZooKeeperMap}
import ru.yandex.vertis.scheduler.builder.Builders
import ru.yandex.vertis.scheduler.model.{Payload, Task, TaskDescriptor}

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 08.09.15
 */
trait SchedulerTasks {
  def schedulingTasks: Seq[Task] = Seq.empty[Task]
}

trait SchedulerSupport {
  app: Application with ZookeeperSupport with HealthChecksSupport with SchedulerTasks =>

  private def zooKeeperBasePath = "/scheduler/" + app.componentName
  private def schedulerInstanceId = app.componentName + "@" + app.hostName

  private lazy val schedulerBuilder = Builders.newZkSchedulerBuilder()
    .setCuratorFramework(zkClient)
    .setZooKeeperBasePath(zooKeeperBasePath)
    .setSchedulerInstanceId(schedulerInstanceId)
    .setMaxConcurrentTasks(16)
    .setMaxRunningWeight(100)
    .setHealthChecks(compoundChecksRegistry)
    .register(schedulingTasks.map(asSwitchable))

  lazy val scheduler = schedulerBuilder.build()

  lazy val taskIsEnabledMap = new ZooKeeperMap[Boolean](
    zkClient, "/scheduler/jobs-config",
    zoo.BooleanSerializer,
    startOnCreate = false,
    syncPeriod = Some(1.minute)
  )() with LoggingZooKeeperMap[Boolean]

  onStart {
    AfterConnectionEstablished.execute(zkClient, new Runnable {
      override def run(): Unit = {
        taskIsEnabledMap.start()
        scheduler.start()

        scheduler.getApi.list().get.foreach {
          ctx ⇒ taskIsEnabledMap.putIfAbsent(nameFor(ctx.descriptor), true, ephemeral = false)
        }
      }
    })
  }

  onStop {
    scheduler.shutdown(1.minute)
  }

  private def asSwitchable(task: Task) = Task(task.descriptor, runIfEnabled(task))
  private def nameFor(descriptor: TaskDescriptor): String = descriptor.id + "@" + componentName
  private def isEnabled(task: Task) = {
    taskIsEnabledMap.snapshot.getOrElse(nameFor(task.descriptor), true)
  }


  private def runIfEnabled(task: Task): Payload = {
    def logInfo() = log.warn(s"task ${task.descriptor.id} is disabled, skipping task run")

    task.payload match {
      case Payload.Async(action) =>
        Payload.Async(() => if (isEnabled(task)) action() else Future.successful(logInfo()))
      case Payload.Sync(action) =>
        Payload.Sync(() => if (isEnabled(task)) action() else logInfo())
      case Payload.SyncTry(action) =>
        Payload.SyncTry(() => if (isEnabled(task)) action() else Try(logInfo()))
    }
  }
}