package ru.yandex.tours.app

import java.net.InetSocketAddress

import akka.actor.Props
import akka.io.{Tcp, IO}
import akka.pattern._
import akka.util.Timeout
import ru.yandex.common.stats.IsTerminatedHealthCheck
import ru.yandex.tours.util.spray.{PingControl, RootHandler}
import spray.can.Http

import scala.concurrent.Await
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 21.01.15
 */
trait SpraySupport {
  this: Application with AkkaSupport with HealthChecksSupport =>

  protected def startServer(handlerProps: Props,
                            port: Int,
                            pingControl: PingControl = new PingControl) {
    implicit val timeout = Timeout(10.seconds)

    var props = RootHandler.props(handlerProps, pingControl)
    if (akkaSystem.dispatchers.hasDispatcher("akka.actor.handler-dispatcher")) {
      props = props.withDispatcher("akka.actor.handler-dispatcher")
    }

    val rootHandler = akkaSystem.actorOf(props, "http-handler")
    val checkName = rootHandler.path.toString
    healthChecksRegistry.unregister(checkName)
    healthChecksRegistry.register(checkName, new IsTerminatedHealthCheck(checkName, rootHandler))

    val soOptions = List(
      Tcp.SO.TcpNoDelay(on = true),
      Tcp.SO.KeepAlive(on = false),
      Tcp.SO.ReuseAddress(on = true)
    )

    val binding = IO(Http)(akkaSystem) ? Http.Bind(
      listener = rootHandler,
      endpoint = new InetSocketAddress(port),
      backlog = 1024,
      options = soOptions,
      settings = None)

    Await.result(binding, timeout.duration) match {
      case Http.Bound(address) =>
        log.info(s"API handler started on $address")
      case msg =>
        log.error(s"Unexpected answer from IO: $msg. Exiting...")
        System.exit(1)
    }
  }
}
