package ru.yandex.tours.app

import org.apache.curator.framework.{CuratorFramework, CuratorFrameworkFactory}
import org.apache.curator.retry.BoundedExponentialBackoffRetry
import ru.yandex.tours.util.Monitorings
import ru.yandex.vertis.curator.monitoring.CuratorMonitoring

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.03.15
 */
trait ZookeeperSupport { this: Application =>
  def zkConnectString: String
  def zkClient: CuratorFramework
  def zkClientWithComponentNamespace: CuratorFramework
}

trait DefaultZookeeperSupport extends ZookeeperSupport { app: Application =>
  override lazy val zkConnectString = app.config.getString("tours.zookeeper.connect-string")

  lazy val zkRetryPolicy = new BoundedExponentialBackoffRetry(10000, 60000, 29)

  override lazy val zkClient =
    CuratorFrameworkFactory.builder()
      .namespace(app.serviceName)
      .connectString(zkConnectString)
      .retryPolicy(zkRetryPolicy)
      .build()

  override lazy val zkClientWithComponentNamespace: CuratorFramework =
    zkClient.usingNamespace(zkClient.getNamespace + "/" + app.componentName)

  onStart {
    zkClient.start()
    log.info("Zookeeper client started")
    CuratorMonitoring.scheduleReconnect(zkClient)
    Monitorings.registry.register("zookeeper",
      CuratorMonitoring.connectionStateHealthCheck("default", zkClient, errorDelay = 5.minutes)
    )
  }

  onStop {
    zkClient.close()
    log.info("Zookeeper client stopped")
  }
}