package ru.yandex.tours.avia

import org.joda.time.DateTimeZone
import ru.yandex.tours.model.LocalizedString
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.parsing.{AtSKV, IntValue, Tabbed}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.02.16
 */
case class Airport(name: LocalizedString,
                   cityName: LocalizedString,
                   geoId: Option[Int],
                   aviaCity: Option[String],
                   aviaId: String,
                   iata: Option[String],
                   timeZone: Option[DateTimeZone]) {

  require(aviaCity.forall(_.startsWith("c")), "aviaCity should starts with 'c'")
  require(aviaId.startsWith("s"), "aviaId should starts with 's'")

  def id: String = aviaCity.getOrElse(aviaId)
}

object Airport {
  def unapply(line: String): Option[Airport] = {
    line match {
      case Tabbed(AtSKV(name), AtSKV(cityName), geoId, aviaCity, aviaId, iata, tz) =>
        Some(Airport(
          LocalizedString.fromMap(name),
          LocalizedString.fromMap(cityName),
          IntValue(geoId),
          optional(aviaCity.nonEmpty, aviaCity),
          aviaId,
          optional(iata.nonEmpty, iata),
          if (tz.nonEmpty) Some(DateTimeZone.forID(tz)) else None
        ))
      case _ =>
        None
    }
  }
}