package ru.yandex.tours.avia

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.IO

class Airports(airports: Iterable[Airport]) {
  private val byIataMap = (for {
    airport <- airports
    iata <- airport.iata.toIterable
  } yield iata -> airport).toMap
  private val byAviaIdMap = airports.map(a => a.aviaId -> a).toMap
  private val byAviaCityIdMap = airports.flatMap(a => a.aviaCity.map(_ -> a)).toMultiMap
  private val byGeoIdMap = airports.flatMap(a => a.geoId.map(_ -> a)).toMultiMap

  def byGeoId(geoId: Int): Seq[Airport] = byGeoIdMap.getOrElse(geoId, Seq.empty)
  def byIata(iata: String): Option[Airport] = byIataMap.get(iata)
  def byAviaId(aviaId: String): Option[Airport] = byAviaIdMap.get(aviaId)
  def byAviaCityId(aviaCityId: String): Seq[Airport] = byAviaCityIdMap.getOrElse(aviaCityId, Seq.empty)

  def byId(id: String): Option[Airport] = byAviaId(id).orElse(byAviaCityId(id).headOption).orElse(byIata(id))

  def size: Int = byAviaIdMap.size
  override def toString: String = s"Airports($size airports)"
}

object Airports extends DataDef[Airports] {
  override def dataType: DataType = DataTypes.aviaAirports

  override def parse(is: InputStream): Airports = {
    val airports = IO.readLines(is).flatMap(l => Airport.unapply(l)).toList
    new Airports(airports)
  }

  def empty: Airports = new Airports(Seq.empty)
}
