package ru.yandex.tours.avia

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.avia.AviaAirportRecommendations.AirportRecommendation
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.util.lang.SideEffectOption
import ru.yandex.tours.util.{Logging, IO}
import ru.yandex.tours.util.parsing.{DoubleValue, IntValue, Tabbed}
import shapeless._

import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 16.09.15
 */
class AviaAirportRecommendations(map: Map[Int, Seq[AirportRecommendation]], tree: region.Tree) extends Serializable {
  private def isFromInsideAirportRegion(from: Int): (AirportRecommendation) => Boolean = {
    val parents = tree.findParents(from).map(_.id)
    (recommendation: AirportRecommendation) => recommendation.city.geoId.exists(parents.contains)
  }

  def recommend(geoId: Int): Seq[AirportRecommendation] = {
    tree.pathToRoot(geoId).iterator
      .map(r => map.getOrElse(r.id, Seq.empty))
      .find(_.nonEmpty)
      .getOrElse(Seq.empty)
  }

  def recommend(from: Int, geoId: Int): Option[AirportRecommendation] = {
    recommendMany(from, geoId).headOption
  }

  def recommendMany(from: Int, to: Int): Seq[AirportRecommendation] = {
    val isTooClose = isFromInsideAirportRegion(from)
    val rec = recommend(to)
    if (rec.headOption.exists(isTooClose)) Seq.empty
    else rec.filterNot(isTooClose)
  }

  def size: Int = map.size
}

object AviaAirportRecommendations
  extends DataDefWithDependencies[AviaAirportRecommendations, region.Tree :: AviaCities :: HNil] with Logging {

  case class AirportRecommendation(city: AviaCity, distance: Option[Double])

  override def dataType: DataType = DataTypes.airportRecommendations
  override def dependsOn: Set[DataType] = Set(DataTypes.aviaCities)

  def empty: AviaAirportRecommendations = new AviaAirportRecommendations(Map.empty, region.Tree.empty)

  override def parse(is: InputStream, dependencies: region.Tree :: AviaCities :: HNil): AviaAirportRecommendations = {
    val tree :: cities :: HNil = dependencies

    val unknownCities = new mutable.HashSet[String]()

    def parseRecommendation(rawAirport: String): Option[AirportRecommendation] = {
      rawAirport.split(":") match {
        case Array(cityId, _) =>
          cities.byId(cityId)
            .onEmpty { unknownCities += cityId }
            .map { city => AirportRecommendation(city, None) }
        case Array(_, _, cityId, _, DoubleValue(distance), _) =>
          cities.byId(cityId)
            .onEmpty { unknownCities += cityId }
            .map { city => AirportRecommendation(city, Some(distance)) }
      }
    }

    val map = Map.newBuilder[Int, Seq[AirportRecommendation]]
    for (Tabbed(IntValue(geoId), _, _, _, _, _, _, recommendations) <- IO.readLines(is)) {
      val airportRecommendations = for {
        raw <- recommendations.split("@")
        recommendation <- parseRecommendation(raw)
      } yield recommendation

      map += geoId -> airportRecommendations.toSeq
    }

    log.warn(s"Found ${unknownCities.size} unknown avia cities in airport_recommendations: " +
      unknownCities.mkString(", "))

    new AviaAirportRecommendations(map.result(), tree)
  }
}
