package ru.yandex.tours.avia

import akka.util.Timeout
import org.joda.time.LocalDate
import play.api.libs.json.Json
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.{StatusCodes, Uri}

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 15.01.16
 */
class AviaCacheClient(httpClient: AsyncHttpClient, aviaAirports: AviaAirportRecommendations)
                     (implicit ec: ExecutionContext) extends Logging {

  private val baseUri = Uri("https://avia.yandex.ru/flexible/")
  private implicit val timeout: Timeout = Timeout(10.seconds)

  def getMinPrice(request: HotelSearchRequest): Future[Option[Int]] = {
    val adults = request.ages.count(_ >= 12)
    val children = request.ages.count(a => a < 12 && a >= 2)
    val infants = request.ages.count(_ < 2)

    aviaAirports.recommend(request.from, request.to) match {
      case Some(recommendation) =>
        val airportId = recommendation.city.id

        val query = Uri.Query(
          "lang" -> "ru",
          "klass" -> "economy",
          "add_today" -> "true",
          "adult_seats" -> adults.toString,
          "children_seats" -> children.toString,
          "infant_seats" -> infants.toString,
          "fromId" -> ("g" + request.from.toString),
          "toId" -> airportId,
          "when" -> request.when.toString,
          "return_date" -> request.when.plusDays(request.nights).toString
        )

        httpClient.get(baseUri.withQuery(query)).map {
          case (StatusCodes.OK, content) =>
            val json = Json.parse(content)
            json.as[AviaCacheResponse].data
              .find(_.when == request.when)
              .flatMap(_.price)
              .map(_.value)
          case (StatusCodes.NotFound, _) =>
            log.warn(s"Cannot search avia prices for region ${request.to}. Tried airport with id = $airportId")
            None
          case (code, content) =>
            sys.error(s"Got error from avia cache (code = $code): $content")
        }.recover {
          case NonFatal(t) =>
            log.warn(s"Failed to load prices from avia cache for request $request", t)
            None
        }
      case None =>
        Future.successful(None)
    }
  }

  private case class Price(currency: String, value: Int)
  private case class Item(url: String, price: Option[Price], when: LocalDate)
  private case class AviaCacheResponse(data: Seq[Item])

  private implicit val priceReads = Json.reads[Price]
  private implicit val itemReads = Json.reads[Item]
  private implicit val responseReads = Json.reads[AviaCacheResponse]
}
