package ru.yandex.tours.avia

import java.util.concurrent.TimeUnit

import org.joda.time.DateTime
import play.api.libs.functional.syntax._
import play.api.libs.json._
import ru.yandex.tours.model.search.FlightSearchRequest

import scala.concurrent.Future
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.02.16
 */
trait AviaClient {

  def getMinPrice(request: FlightSearchRequest, update: Boolean): Future[AviaClient.Response]

  def getMinPriceFromRasp(request: FlightSearchRequest): Future[AviaClient.RaspResponse]

  def getMinPriceUpdated(searchId: String): Future[AviaClient.ResponseUpdated]

  def getDictionaries: Future[AviaClient.Dictionaries]
}

object AviaClient {
  case class Session(deadline: Deadline, xSession: String) {
    def headers: List[(String, String)] = List("X-SESSION" -> xSession)
  }

  case class Tariff(currency: String, price: Int)
  case class FlightVariant(forward: Seq[String], backward: Seq[String]) {
    def isUseful: Boolean = forward.nonEmpty && backward.nonEmpty
  }
  case class MinPrice(variants: Seq[FlightVariant], tariff: Tariff) {
    def isUseful: Boolean = variants.nonEmpty && variants.forall(_.isUseful)
  }
  case class Flight(key: String,
                    number: String,
                    departure_key: String, arrival_key: String,
                    company_id: Int,
                    path_duration: Int,
                    departure_datetime: DateTime, arrival_datetime: DateTime)
  case class Company(id: Int, sirena_id: Option[String], iata: Option[String], title: String, iata_priority: Int)
  case class MinPrices(indirect: Option[MinPrice], direct: Option[MinPrice],
                       flights: Seq[Flight], companies: Seq[Company]) {
    def isUseful: Boolean = direct.forall(_.isUseful) && indirect.forall(_.isUseful)
    def isEmpty: Boolean = indirect.isEmpty && direct.isEmpty
    def nonEmpty: Boolean = !isEmpty
  }
  case class Response(minPrices: MinPrices, search_id: Option[String]) {
    def isUseful: Boolean = minPrices.isUseful
  }
  case class ResponseUpdated(result: Option[MinPrices], status: String)

  case class City(key: String, title: String, country_key: String,
                  geoid: Option[Int], iata: Option[String], sirena: Option[String])
  case class Country(key: String, title: String, geoid: Option[Int], iso_code: Option[String])
  case class Airport(key: String, title: String, city_key: Option[String], iata: Option[String], sirena: Option[String])

  case class Dictionaries(cities: Seq[City], countries: Seq[Country], airports: Seq[Airport])

  private implicit val dateTimeReads = new Reads[DateTime] {
    override def reads(json: JsValue): JsResult[DateTime] = JsSuccess(DateTime.parse(json.as[String]))
  }
  private implicit val tariffReads = Json.reads[Tariff]
  private implicit val flightVariantReads = (
    (__ \ "forward").readNullable[Seq[String]].map(_.toSeq.flatten) and
    (__ \ "backward").readNullable[Seq[String]].map(_.toSeq.flatten)
  )(FlightVariant)
  private implicit val minPriceReads = Json.reads[MinPrice]
  private implicit val flightReads = Json.reads[Flight]
  private implicit val companyReads = Json.reads[Company]
  private implicit val minPricesReads: Reads[MinPrices] = (
    (__ \\ "indirect").readNullable[MinPrice] and
    (__ \\ "direct").readNullable[MinPrice] and
    (__ \ "flights").read[Seq[Flight]] and
    (__ \ "companies").read[Seq[Company]]
  )(MinPrices)
  implicit val responseReads: Reads[Response] = (
    (__ \ "data").read[MinPrices] and
    (__ \ "data" \ "search_id").readNullable[String]
  )(Response)
  implicit val responseUpdatedReads: Reads[ResponseUpdated] = (
    (__ \ "data" \ "result").readNullable[MinPrices] and
    (__ \ "data" \ "status").read[String]
  )(ResponseUpdated)
  implicit val sessionReads: Reads[Session] = (
    (__ \ "data" \ "ttl_seconds").read[Int].map(s => Duration.apply(s, TimeUnit.SECONDS).fromNow) and
    (__ \ "data" \ "X-SESSION").read[String]
  )(Session)
  private implicit val cityReads = Json.reads[City]
  private implicit val countryReads = Json.reads[Country]
  private implicit val airportReads = Json.reads[Airport]
  implicit val dictionariesReads: Reads[Dictionaries] = (
    (__ \ "data" \ "cities").read[Seq[City]] and
    (__ \ "data" \ "countries").read[Seq[Country]] and
    (__ \ "data" \ "airports").read[Seq[Airport]]
  )(Dictionaries)

  case class RaspCompany(alliance: Option[Int], title: String, url: String, id: Int)
  case class RaspTime(local: String, tzname: String)
  case class RaspFlight(arrival: RaspTime, departure: RaspTime, to: Int, from: Int,
                        key: String, company: Int, number: String, url: String, companyTariff: Int)
  case class RaspTariff(currency: String, value: Int)
  case class RaspPrice(boy: Boolean, tariff_sign: String, queryTime: Int, baggage: Seq[Seq[String]],
                       partnerCode: String, tariff: RaspTariff)
  case class RaspFare(prices: Seq[RaspPrice], route: Seq[Seq[String]], hybrid: Boolean) {
    def correct: Boolean = route.length == 2 && route.last.length<=1 && route.head.length==1 && prices.forall(_.tariff.currency == "RUR")
    def forward: Option[String] = route.toList.head.headOption
    def backward: Option[String] = route.toList.apply(1).headOption

    def totalPrice: Int = {
      prices.filter(_.tariff.currency=="RUR").foldLeft(0)((m, mp) => mp.tariff.value + m)
    }
  }
  case class RaspResponse(status: String,
                          fares: Seq[RaspFare],
                          flights: Seq[RaspFlight],
                          companies: Seq[RaspCompany]
                          )
  private implicit val raspTimeReads = Json.reads[RaspTime]
  private implicit val raspFlightReads = Json.reads[RaspFlight]
  private implicit val raspTariffReads = Json.reads[RaspTariff]
  private implicit val raspPriceReads = Json.reads[RaspPrice]
  private implicit val raspFareReads = Json.reads[RaspFare]
  private implicit val raspCompReads = Json.reads[RaspCompany]
  implicit val raspResponseReads: Reads[RaspResponse] = (
    (__ \ "status").read[String] and
    (__ \ "data" \ "variants" \ "fares").read[Seq[RaspFare]] and
    (__ \ "data" \ "reference" \ "flights").read[Seq[RaspFlight]] and
    (__ \ "data" \ "reference" \ "companies").read[Seq[RaspCompany]]
  )(RaspResponse)
}