package ru.yandex.tours.avia

import java.util.concurrent.atomic.AtomicReference

import akka.util.Timeout
import org.joda.time.{DateTime, DateTimeZone}
import play.api.libs.json.Json
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.03.16
 */
class DefaultAviaClient(httpClient: AsyncHttpClient,
                        raspUri: Uri,
                        baseUri: Uri,
                        appKey: String,
                        oauthToken: String)
                       (implicit ec: ExecutionContext) extends AviaClient with Logging {

  import AviaClient._

  implicit val timeout = Timeout(60.seconds)

  private val raspMinPrices = Uri(raspUri + "/jsendapi/v3/wizard_results")
  private val uuid = "bcaff0f8fa294622b2367af2ccd1fdbe"
  private val pushToken = "ya.travel"

  private val hello = Uri(baseUri + "/hello/")
  private val minPrices = Uri(baseUri + "/min_prices/")
  private val minPricesUpdated = Uri(baseUri + "/min_prices/updated/")
  private val dictionariesUri = Uri(baseUri + "/info/por/")

  private val lastSession = new AtomicReference[Future[Session]](getNewSession)

  private def getNewSession: Future[Session] = {
    val uri = hello.withQuery(
      "app_key" -> appKey,
      "oauth_token" -> oauthToken,
      "timestamp" -> DateTime.now().withZone(DateTimeZone.UTC).toString("yyyy-MM-dd HH:mm:ss"),
      "uuid" -> uuid,
      "push_token" -> pushToken,
      "national_version" -> "ru"
    )
    log.info("Refreshing session")
    httpClient.post(uri, Array.emptyByteArray).map {
      case (StatusCodes.OK, body) =>
        Json.parse(body).as[Session]
      case (sc, body) =>
        val msg = s"Unexpected response $sc from avia hello method. Uri = $uri, Body = $body"
        log.error(msg)
        sys.error(msg)
    }
  }

  private def getSession: Future[Session] = {
    val current = lastSession.get
    def reload() = {
      val promise = Promise.apply[Session]()
      if (lastSession.compareAndSet(current, promise.future)) {
        promise.completeWith(getNewSession)
        promise.future
      } else {
        getSession
      }
    }

    current.recoverWith {
      case NonFatal(t) => reload()
    }.flatMap { session =>
      if (session.deadline.isOverdue()) {
        reload()
      } else {
        Future.successful(session)
      }
    }
  }

  def getMinPriceFromRasp(request: FlightSearchRequest): Future[AviaClient.RaspResponse] = {
    val adults = request.hotelRequest.ages.count(_ >= 12)
    val children = request.hotelRequest.ages.count(a => a < 12 && a >= 2)
    val infants = request.hotelRequest.ages.count(_ < 2)
    val airportId = request.airportId

    val uri = raspMinPrices.withQuery(
      "lang" -> "ru",
      "mobile" -> "false",
      "national" -> "ru",
      "platform" -> "web",
      "service" -> "tours",
      "ignoreOutdated" -> "true",
      "klass" -> "economy",
      "adults" -> adults.toString,
      "children" -> children.toString,
      "infants" -> infants.toString,
      "date_forward" -> request.hotelRequest.when.toString,
      "date_backward" -> request.hotelRequest.whenBack.toString,
      "point_from" -> ("c" + request.hotelRequest.from),
      "point_to" -> airportId
    )
    for ((sc, body) <- httpClient.get(uri))
      yield {
        if (sc != StatusCodes.OK) {
          sys.error(s"Unexpected response $sc from avia rasp min_prices method. Uri = $uri, Body = $body")
        }
        try Json.parse(body).as[AviaClient.RaspResponse]
        catch {
          case NonFatal(t) =>
            throw new RuntimeException(s"Failed to parse avia rasp response from Uri = $uri, Body = $body", t)
        }
      }
  }

  def getMinPrice(request: FlightSearchRequest, update: Boolean): Future[AviaClient.Response] = {
    val adults = request.hotelRequest.ages.count(_ >= 12)
    val children = request.hotelRequest.ages.count(a => a < 12 && a >= 2)
    val infants = request.hotelRequest.ages.count(_ < 2)
    val airportId = request.airportId

    val uri = minPrices.withQuery(
      "date_range" -> "0",
      "lang" -> "ru",
      "adults" -> adults.toString,
      "children" -> children.toString,
      "infants" -> infants.toString,
      "when" -> request.hotelRequest.when.toString,
      "return_date" -> request.hotelRequest.whenBack.toString,
      "from" -> ("g" + request.hotelRequest.from),
      "to" -> airportId,
      "update_cache" -> update.toString
    )
    for {
      session <- getSession
      (sc, body) <- httpClient.get(uri, session.headers)
    } yield {
      if (sc != StatusCodes.OK) {
        sys.error(s"Unexpected response $sc from avia /min_prices method. Uri = $uri, Body = $body")
      }
      try Json.parse(body).as[AviaClient.Response]
      catch {
        case NonFatal(t) =>
          throw new RuntimeException(s"Failed to parse avia response from Uri = $uri, Body = $body", t)
      }
    }
  }

  def getMinPriceUpdated(searchId: String): Future[AviaClient.ResponseUpdated] = {
    val uri = minPricesUpdated.withQuery(
      "search_id" -> searchId,
      "lang" -> "ru"
    )
    for {
      session <- getSession
      (sc, body) <- httpClient.get(uri, session.headers)
    } yield {
      if (sc != StatusCodes.OK) {
        sys.error(s"Unexpected response $sc from avia /min_prices/updated method. Uri = $uri, Body = $body")
      }
      try Json.parse(body).as[AviaClient.ResponseUpdated]
      catch {
        case NonFatal(t) =>
          throw new RuntimeException(s"Failed to parse avia response from Uri = $uri, Body = $body", t)
      }
    }
  }

  def getDictionaries: Future[AviaClient.Dictionaries] = {
    val uri = dictionariesUri.withQuery("lang" -> "ru")
    for {
      session <- getSession
      (StatusCodes.OK, body) <- httpClient.get(uri, session.headers)
    } yield {
      try Json.parse(body).as[AviaClient.Dictionaries]
      catch {
        case NonFatal(t) =>
          throw new RuntimeException(s"Failed to parse avia response from Uri = $uri, Body = $body", t)
      }
    }
  }
}