package ru.yandex.tours.billing

import java.io.InputStream

import com.codahale.metrics.health.HealthCheck
import com.codahale.metrics.health.HealthCheck.Result
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.billing.BillingOffers.IdedOfferBilling
import ru.yandex.tours.extdata.DataDefWithDependencies
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.util.{Logging, Monitorings}
import ru.yandex.vertis.billing.Model.OfferBilling
import shapeless.HList

trait BillingIndexTrait[D <: HList] extends DataDefWithDependencies[BillingIndex, D] {

  override def dependsOn: Set[DataType] = Set.empty

  override def parse(is: InputStream, dependencies: D): BillingIndex = {
    val offers = BillingOffers.parseIdAndOffers(is).filter {
      case (id, offer) => offer.hasKnownCampaign && offer.getKnownCampaign.getIsActive
    }
    val map = offers.map {
      case (id, offer) => id -> createIdedOffer(id, offer)
    }.toMap
    createIndex(map, dependencies)
  }

  protected def createIdedOffer(id: Long, offer: OfferBilling): IdedOfferBilling

  protected def createIndex(map: Map[Long, IdedOfferBilling], dependencies: D) = new BillingIndex(map)

  def monitor(index: BillingIndex): Unit = {
    Monitorings.registry.registerDeveloper(dataType.getName, new HealthCheck {
      override def check(): Result = {
        val activeCount = index.allActive.size
        val message = s"${dataType.getName} billing has $activeCount active campaigns"
        if (activeCount == 0) {
          Result.unhealthy(message)
        } else {
          Result.healthy(message)
        }
      }
    })
  }
}

class BillingIndex(val id2offer: Map[Long, IdedOfferBilling]) extends Logging {

  def allActive: Iterable[IdedOfferBilling] = id2offer.values.filter(_.isActive)

  def getOfferBilling(id: Long): Option[IdedOfferBilling] = id2offer.get(id)

  def isEnabled(id: Long): Boolean = {
    getOfferBilling(id) match {
      case Some(offerBilling) =>
        val campaignHeader = offerBilling.offer.getKnownCampaign.getCampaign
        BillingServiceCommon.isEnabled(campaignHeader)
      case None => false
    }
  }

  def filterOffers(partner: Partner, offers: Iterable[Offer]): Iterable[Offer] = {
    val filtered = offers.filter { offer => isEnabled(offer.getHotelId) }
    log.info(s"Filtered channel manager hotels for partner: $partner. " +
      s"Found offers: ${offers.size}. " +
      s"Billing enabled: ${filtered.size}.")
    filtered
  }
}
