package ru.yandex.tours.billing

import java.io.{DataInputStream, EOFException, InputStream}

import org.apache.commons.io.IOUtils
import org.joda.time.{DateTime, DateTimeZone}
import ru.yandex.tours.model.purchase.BillingObject
import ru.yandex.vertis.billing.Model.{CampaignHeader, OfferBilling}
import ru.yandex.vertis.billing.model.Properties
import ru.yandex.vertis.protobuf.kv.Converter

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.util.Try

object BillingOffers {
  private def readLong(dis: DataInputStream) = {
    try {
      Some(dis.readLong())
    } catch {
      case e: EOFException => None
    }
  }

  def parseIdAndOffers(is: InputStream): Iterable[(Long, OfferBilling)] = {
    try {
      val dis = new DataInputStream(is)
      var objects = Vector.newBuilder[(Long, OfferBilling)]
      var currentId = readLong(dis)
      var offer = OfferBilling.parseDelimitedFrom(is)
      while (currentId.isDefined && offer != null) {
        objects += currentId.get -> offer
        currentId = readLong(dis)
        offer = OfferBilling.parseDelimitedFrom(is)
      }
      objects.result()
    } finally {
      IOUtils.closeQuietly(is)
    }
  }

  def dumpOffer(idedOffer: IdedOfferBilling): Try[mutable.Map[String, String]] = {
    val id = idedOffer.billingId
    val offer = idedOffer.offer
    Converter.toKeyValue(offer.getKnownCampaign.getCampaign, Some(Properties.BILLING_CAMPAIGN_HEADER)).map { map =>
      val deadline = new DateTime(offer.getKnownCampaign.getActiveDeadline, DateTimeZone.UTC).toString
      Properties.getCommon(deadline, id.toString) ++ map ++ Iterable("offer_partner_id" -> id, "business_id" -> id)
    }
  }
  
  case class IdedOfferBilling(billingId: String, offer: OfferBilling) {
    def isActive: Boolean = offer.getKnownCampaign.getActiveDeadline > System.currentTimeMillis
  }
}

object CMBillingOffers {
  import ru.yandex.tours.billing.BillingOffers._

  def toBillingObject(idedOffer: IdedOfferBilling): Try[BillingObject] = {
    for {
      uiInfo <- dumpOffer(idedOffer)
    } yield {
      uiInfo += "offer_id" -> idedOffer.billingId
      val campaignHeader = idedOffer.offer.getKnownCampaign.getCampaign
      val clickRevenueOpt = getClickRevenue(campaignHeader)
      BillingObject(campaignHeader.getId, uiInfo, clickRevenueOpt)
    }
  }

  private def getClickRevenue(header: CampaignHeader): Option[Long] = Try {
    header.getProduct.getGoods(0).getPlacement.getCost.getPerClick.getUnits
  }.toOption
}