package ru.yandex.tours.billing

import java.util.concurrent.atomic.AtomicReference

import akka.actor.ActorSystem
import org.joda.time.DateTime
import ru.yandex.tours.model.purchase.BillingObject
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.{Logging, Monitorings}
import ru.yandex.vertis.billing.CampaignsClient
import ru.yandex.vertis.billing.model.Properties
import ru.yandex.vertis.protobuf.kv.Converter

import scala.collection.JavaConversions._
import scala.concurrent.ExecutionContext
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

object HttpBillingService extends Logging{
  val UPDATE_INTERVAL: FiniteDuration = 1.minute
  private val VALID_PERIOD_IN_HOUR = 2
}

class HttpBillingService(client: CampaignsClient)(implicit as: ActorSystem, ec: ExecutionContext)
  extends BillingService with Logging {

  import ru.yandex.tours.billing.HttpBillingService._

  private val activePartners = new AtomicReference[Map[String, BillingObject]](Map.empty)

  private val updateErrorMonitoring = Monitorings("http-billing-client")
    .errorCount("update", 3, 20, 1.hour, 60)

  def update(): Try[Map[String, BillingObject]] = {
    client.getCampaignHeaders.map(headers => {
      val partners = headers.filter(isEnabled).flatMap(header => {
        val refs = header.getOwner.getResourceRefList
        val res = for {
          uiInfo <- Converter.toKeyValue(header, Some(Properties.BILLING_CAMPAIGN_HEADER))
          filteredRefs = refs.filter(_.hasCapaPartnerId)
          id <- {
            if (filteredRefs.size == 1) Success(filteredRefs.head.getCapaPartnerId)
            else Failure(new Exception("refs with capa id count should be 1"))
          }
        } yield {
          val clickRevenue = if (header.getProduct.getGoodsCount > 0) {
            val cost = header.getProduct.getGoods(0).getPlacement.getCost
            if (cost.hasPerClick) {
              Some(cost.getPerClick.getUnits)
            } else {
              None
            }
          } else {
            None
          }
          id -> BillingObject(header.getId, uiInfo, clickRevenue)
        }
        res onFailure {
          e => log.warn(s"Can not parse billing header with id = ${header.getId}", e)
        }
        res.toOption
      }).toMap
      activePartners.set(partners)
      updateErrorMonitoring.ok()
      log.info("Billing headers successfully updated. " +
        s"${partners.size} active campaigns: ${partners.keys.mkString(", ")}")
      if (log.isDebugEnabled) {
        partners.foreach { case (partner, billing) =>
          log.debug(s"Billing for [$partner]: $billing")
        }
      }
      partners
    }) onFailure {
      e =>
        updateErrorMonitoring.error(e)
        log.error("Can not update billing info", e)
    }
  }

  as.scheduler.schedule(0.minutes, UPDATE_INTERVAL, new Runnable {
    override def run(): Unit = update()
  })

  override def isActive(partnerCode: String): Boolean = activePartners.get().contains(partnerCode)

  override def getBillingInfo(partnerCode: String, offerId: String): Option[BillingObject] = {
    val deadline = DateTime.now.plusHours(VALID_PERIOD_IN_HOUR).toString
    activePartners.get.get(partnerCode)
      .map { o =>
        o.copy(
          uiInfo = o.uiInfo ++
            Iterable("offer_id" -> offerId, "offer_partner_id" -> partnerCode) ++
            Properties.getCommon(deadline, offerId)
        )
      }
  }
}
