package ru.yandex.tours.calendar

import ru.yandex.tours.model.Prices.PriceGraph
import ru.yandex.tours.model.Prices.PriceGraph.PricedOperator
import ru.yandex.tours.model.Source
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.services.{CalendarService, MinPriceService}
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.http.HttpTimeout
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.06.15
 */
object CalendarUtils {
  private def mergeMinPrices[S <: Source](prices: Seq[PricedOperator], searchSources: SearchSources[S]): Int = {
    if (prices.isEmpty) MinPriceService.UNKNOWN_PRICE
    else {
      val withPrices = prices.filter(_.getPrice > 0)
      if (withPrices.isEmpty) {
        if (searchSources.getAll.size == prices.size) MinPriceService.NO_TOURS else MinPriceService.UNKNOWN_PRICE
      } else {
        withPrices.map(_.getPrice).min
      }
    }
  }

  private def noTours(operatorId: Int) = {
    PricedOperator.newBuilder()
      .setOperatorId(operatorId)
      .setPrice(MinPriceService.NO_TOURS)
      .build
  }

  def mergeGraphWithCalendar[S <: Source](graph: PriceGraph,
                                          calendarService: CalendarService,
                                          operators: SearchSources[S])
                                         (implicit ec: ExecutionContext): Future[PriceGraph] = {

    val result = calendarService.getNoFlights(graph.getFrom, graph.getTo, Some(graph.getNights))
      .recover { case HttpTimeout(_) ⇒ Seq.empty }
    for (result <- result) yield {
      val noFlights = result
        .filter(_.getNights == graph.getNights)
        .map { entry => entry.getWhen.toLocalDate -> SimpleBitSet.from(entry.getNoDeparture).toSet }
        .toMap.withDefaultValue(Set.empty)

      val graphWhen = graph.getWhen.toLocalDate
      val graphBuilder = graph.toBuilder
      for (i <- 0 until graph.getPriceEntityCount) {
        val entity = graph.getPriceEntity(i)
        val date = graphWhen.plusDays(i - graph.getIndex)
        if (entity.getMinPrice == MinPriceService.UNKNOWN_PRICE) {
          val builder = graphBuilder.getPriceEntityBuilder(i)
          for (operator <- noFlights(date)) {
            builder.addOperatorPrice(noTours(operator))
          }
          builder.setMinPrice(mergeMinPrices(builder.getOperatorPriceList, operators))
        }
      }
      graphBuilder.build()
    }
  }
}
