package ru.yandex.tours.calendar

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import ru.yandex.tours.clickhouse.ClickHouseClient
import ru.yandex.tours.model.Calendar.{Document, FlightDay, FlightMatrix}
import ru.yandex.tours.model.search.SearchDates
import ru.yandex.tours.model.util.{CustomNights, DateInterval, Nights}
import ru.yandex.tours.services.CalendarService
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.parsing.{IntValue, LocalDateValue}
import ru.yandex.tours.util.zoo.SharedValue
import ru.yandex.tours.util.Logging

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 31.03.16
 */
class ClickHouseCalendarService(clickHouseClient: ClickHouseClient,
                                enableReads: Option[SharedValue[Boolean]])
                               (implicit ec: ExecutionContext) extends CalendarService with Logging {


  private val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")

  private def dateIn(column: String, dateInterval: DateInterval) = {
    import dateInterval._
    s"($column >= \'${start.toString(dateFormat)}\') AND ($column <= \'${end.toString(dateFormat)}\')"
  }

  private def nightsIn(column: String, nights: Nights) = {
    if (nights.values.isEmpty) "(1 = 1)"
    else s"($column in ${nights.values.mkString("(", ",", ")")})"
  }

  private def flightsQuery(from: Int, to: Int, dates: SearchDates, noDeparture: Boolean) = {
    val where =
      s"""from = $from
         |AND to = $to
         |AND ages = '88,88'
         |AND ${dateIn("when", dates.when)}
         |AND ${nightsIn("nights", dates.nights)}
         |AND ${dates.whenBack.fold("1 = 1")(dateIn("(when + nights)", _))}
         |AND when >= today()
       """.stripMargin

    val empty =
      s"""SELECT
         |  when, nights, operator_id,
         |  max(timestamp) AS empty_timestamp
         |FROM ${clickHouseClient.database}.tour_empty_results
         |WHERE $where
         |GROUP BY when, nights, operator_id
       """.stripMargin
    val prices =
      s"""SELECT
         |  when, nights, operator_id,
         |  max(timestamp) AS price_timestamp
         |FROM ${clickHouseClient.database}.tour_prices
         |WHERE $where
         |  AND notEmpty(prices.price)
         |GROUP BY when, nights, operator_id
       """.stripMargin

    s"""SELECT
       |  when, nights,
       |  groupUniqArrayIf(operator_id, not no_departure) AS with_dep,
       |  groupUniqArrayIf(operator_id, no_departure) AS no_dep
       |FROM (
       |  SELECT
       |    when, nights,
       |    operator_id,
       |    (price_timestamp < empty_timestamp) AS no_departure
       |  FROM (
       |    ${if (noDeparture) empty else prices}
       |  )
       |  ANY LEFT JOIN (
       |    ${if (!noDeparture) empty else prices}
       |  ) USING when, nights, operator_id
       |)
       |GROUP BY when, nights
       |HAVING
       |  ${if (noDeparture) "notEmpty(no_dep)" else "notEmpty(with_dep)"}
     """.stripMargin
  }

  override def getFlightMatrix: Future[FlightMatrix] = {
    if (enableReads.isDefined && !enableReads.get.get) {
      return Future.successful(FlightMatrix.getDefaultInstance)
    }
    clickHouseClient.query(
      s"""SELECT `from`, to
        |FROM ${clickHouseClient.database}.tour_prices
        |WHERE notEmpty(prices.price)
        |GROUP BY from, to""".stripMargin)
      .map( lines => {
          val builder = FlightMatrix.newBuilder()
          lines.collect {
            case Seq(IntValue(from), IntValue(to)) => from -> to
          }.toVector.toMultiMap.foreach {
            case (from, destinations) =>
              builder.addRowBuilder()
                .setFrom(from)
                .addAllTo(destinations.map(Int.box).asJava)
          }
          builder.build()
        }
      ).recoverWith({
        case exc =>
          log.error(s"Failed to getFlightMatrix: $exc")
          Future.successful(FlightMatrix.getDefaultInstance)
      })
  }

  override def getNearestFlightDay(from: Int, to: Int,
                                   when: LocalDate, interval: DateInterval): Future[Option[FlightDay]] = {
    if (enableReads.isDefined && !enableReads.get.get) {
      return Future.successful(None)
    }

    val q = s"""SELECT
      |  argMin(when, day_diff),
      |  argMin(nights, day_diff)
      |FROM (
      |  SELECT
      |    when,
      |    nights,
      |    timestamp AS price_timestamp,
      |    abs(when - toDate('${when.toString(dateFormat)}')) AS day_diff
      |  FROM ${clickHouseClient.database}.tour_prices AS tp
      |  ANY LEFT JOIN (
      |    SELECT
      |      when, nights, operator_id,
      |      timestamp AS empty_timestamp
      |    FROM ${clickHouseClient.database}.tour_empty_results
      |    WHERE from = $from AND to = $to AND ages = '88,88'
      |  ) USING when, nights, operator_id
      |  WHERE from = $from
      |    AND to = $to
      |    AND ages = '88,88'
      |    AND ${dateIn("when", interval)}
      |    AND when > today()
      |    AND notEmpty(prices.price)
      |    AND price_timestamp > empty_timestamp
      |)
      |GROUP BY 1
       """.stripMargin

    clickHouseClient.query(q)
      .map { lines =>
        lines.headOption.map {
          case Seq(LocalDateValue(w), IntValue(n)) =>
            FlightDay.newBuilder()
              .setWhen(w.toCompactInt)
              .setNights(n)
              .build()
        }
      }.recoverWith({
        case exc =>
          log.error(s"Failed to getNearestFlightDay: $exc")
          Future.successful(None)
    })
  }

  override def getHasFlights(from: Int, to: Int, nights: Option[Int]): Future[Seq[Document]] = {
    if (enableReads.isDefined && !enableReads.get.get) {
      return Future.successful(Seq.empty[Document])
    }
    val now = LocalDate.now()
    val customNights = CustomNights(nights.toSeq)
    val dates = SearchDates(DateInterval(now, now.plusYears(1)), customNights, whenBack = None)

    clickHouseClient.query(flightsQuery(from, to, dates, noDeparture = false))
      .map(parseDocuments(from, to))
      .recoverWith {
        case exc =>
          log.error(s"Failed to getHasFlights: $exc")
          Future.successful(Seq.empty[Document])
      }
  }

  override def getNoFlights(from: Int, to: Int, nights: Option[Int]): Future[Seq[Document]] = {
    if (enableReads.isDefined && !enableReads.get.get) {
      return Future.successful(Seq.empty[Document])
    }
    val now = LocalDate.now()
    val customNights = CustomNights(nights.toSeq)
    val dates = SearchDates(DateInterval(now, now.plusYears(1)), customNights, whenBack = None)

    clickHouseClient.query(flightsQuery(from, to, dates, noDeparture = true))
      .map(parseDocuments(from, to))
      .recoverWith {
        case exc =>
          log.error(s"Failed to getNoFlights: $exc")
          Future.successful(Seq.empty[Document])
      }
  }

  private def parseIntArray(str: String): Array[Int] = {
    str.stripPrefix("[").stripSuffix("]").split(",").filter(_.nonEmpty).map(IntValue.parse)
  }

  private def parseDocuments(from: Int, to: Int)(lines: Iterator[Seq[String]]): Vector[Document] = {
    lines.map {
      case data @ Seq(LocalDateValue(date), IntValue(night), withDeparture, noDeparture) =>
        Document.newBuilder()
          .setFrom(from)
          .setTo(to)
          .setWhen(date.toCompactInt)
          .setNights(night)
          .setHasDeparture(SimpleBitSet(parseIntArray(withDeparture)).packed)
          .setNoDeparture(SimpleBitSet(parseIntArray(noDeparture)).packed)
          .build()
    }.toVector
  }
}
